#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor

BANK_HOSTS = [
    'sberbank.ru',
    'vtb.ru',
    'alfabank.ru',
    'gazprombank.ru',
    'rshb.ru',
    'pochtabank.ru',
    'tinkoff.ru',
    'rosbank.ru',
    'raiffeisen.ru',
    'open.ru',
    'sovcombank.ru',
    'homecredit.ru',
    'unicreditbank.ru',
    'psbank.ru',
    'rencredit.ru',
    'uralsib.ru',
    'rusfinancebank.ru',
    'rsb.ru',
    'cetelem.ru',
    'mkb.ru',
    'vostbank.ru',
    'bspb.ru',
    'sviaz-bank.ru',
    'absolutbank.ru',
    'otpbank.ru',
    'vbank.ru',
    'crediteurope.ru',
    'mtsbank.ru',
    'akbars.ru',
    'zapsibkombank.ru',
    'centrinvest.ru',
    'lockoBank.ru',
    'trust.ru',
    'citibank.ru',
    'zenit.ru',
]

MFO_HOSTS = [
    'moneyman.ru',
    'platiza.ru',
    'ezaem.ru',
    'moneza.ru',
    'eKapusta.com',
    'zaymer.ru',
    'turbozaim.ru',
    'vivus.ru',
    'metrokredit.ru',
    'smsfinance.ru',
    'lime-zaim.ru',
    'webbankir.com',
    '4slovo.ru',
    'joy.money',
    'kredito24.ru',
    'creditplus.ru',
    'Cash-U.com',
    'zaymigo.com',
    'web-zaim.ru',
    'payps.ru',
    'dozarplati.com',
    'ferratum.ru',
    'glavfinans.ru',
    'creditstar.ru',
    'planetacash.ru',
    'Konga.ru',
    'oneclickmoney.ru',
    'migone.ru',
    'creditter.ru',
    'fastmoney.ru',
    'denginadom.ru',
    'bistrodengi.ru',
    'vivadengi.ru',
    'migcredit.ru',
    'cashtoyou.ru',
]

CREDIT_BROCKERS_HOSTS = [
    'credilo.com.ru',
    'visame.com.ru',
    'mazilla.com.ru',
    'nalichka.su',
    'zaimy-bez-otkazov.ru',
    'monebo.ru',
    'hot-zaim.ru',
    'checkmoney.su',
    'good-zaim.su',
    'good-zaem.su',
    'upetrovicha.su',
    'moneykite.ru',
    'banando.ru',
    'mandarino.su',
    'finspin.ru',
    'dengibystra.ru',
    'rublimo.ru',
    'bzaim5.ru',
    'creditkin.su',
    'quickzaim.com',
    'izaime.ru',
    'creditnice.ru',
    'fincash.su',
    'filkos.com',
    'zaim-bez-otkaza.ru',
    'goodzaem.ru',
    'sravnizaim.ru',
    'zaim.com',
]

CREDIT_BROCKERS_URLS = [
    'https://bankiros.ru/zaymy',
    'https://moskva.bankiros.ru/zaymy',
    'https://www.sravni.ru/zaimy/',
    'https://www.sravni.ru/zaimy/moskva/',
    'http://moskva.1000bankov.ru/kredit/zaim/',
    'http://1000bankov.ru/kredit/zaim/',
]

BETTING_HOSTS = [
    'fonbet.ru',
    '1xStavka.ru',
    'parimatch.ru',
    'www.ligastavok.ru',
    'winline.ru',
    'www.marathonbet.ru',
    'betcity.ru',
    'ggbet.ru',
    'olimp.bet',
]

select_clients_query = """
$bank_hosts = AsSet(
{bank_hosts}
);

$mfo_hosts = AsSet(
{mfo_hosts}
);

$broker_hosts = AsSet(
{broker_hosts}
);

$broker_urls = AsSet(
{broker_urls}
);

$betting_hosts = AsSet(
{betting_hosts}
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT *
FROM (
    SELECT
        yandexuid,
        CASE
            WHEN host IN $bank_hosts THEN TRUE
            ELSE NULL
        END AS bank,

        CASE
            WHEN host IN $mfo_hosts THEN TRUE
            ELSE NULL
        END AS mfo,

        CASE
            WHEN host IN $broker_hosts THEN TRUE
            ELSE NULL
        END AS broker_hosts,

        CASE
            WHEN url IN $broker_urls THEN TRUE
            ELSE NULL
        END AS broker_urls,

        CASE
            WHEN host IN $betting_hosts THEN TRUE
            ELSE NULL
        END AS betting,
    FROM `{input_table}`
    WHERE referer LIKE '%yandex.ru/search%' OR referer LIKE '%yandex.ru/clck%'
)
WHERE
    bank IS NOT NULL OR
    mfo IS NOT NULL OR
    broker_hosts IS NOT NULL OR
    broker_urls IS NOT NULL OR
    betting IS NOT NULL;
"""

build_segment_query = u"""
$counts = (
SELECT yandexuid,
       COUNT_IF(bank) AS bank_count,
       COUNT_IF(mfo) AS mfo_count,
       COUNT_IF(broker_hosts OR broker_urls) AS broker_count,
       COUNT_IF(betting) AS betting_count,
FROM `{input_table}`
GROUP BY yandexuid
);


INSERT INTO `{output_table}` WITH TRUNCATE
SELECT CAST(yandexuid AS String) AS id,
       'yandexuid' AS id_type,
       'bank' AS segment_name
FROM $counts
WHERE bank_count > 2

UNION ALL

SELECT CAST(yandexuid AS String) AS id,
       'yandexuid' AS id_type,
       'mfo' AS segment_name
FROM $counts
WHERE mfo_count > 2

UNION ALL

SELECT CAST(yandexuid AS String) AS id,
       'yandexuid' AS id_type,
       'broker' AS segment_name
FROM $counts
WHERE broker_count > 2

UNION ALL

SELECT CAST(yandexuid AS String) AS id,
       'yandexuid' AS id_type,
       'betting' AS segment_name
FROM $counts
WHERE betting_count > 2;
"""


class ProcessedMetricsForProleads(DayProcessor):
    def requires(self):
        return ExternalInput(os.path.join(config.METRICS_PARSED_DIR, self.date))

    def process_day(self, inputs, output_path):
        self.yql.query(
            select_clients_query.format(
                input_table=inputs.table,
                bank_hosts=',\n'.join(['"{}"'.format(host) for host in BANK_HOSTS]),
                mfo_hosts=',\n'.join(['"{}"'.format(host) for host in MFO_HOSTS]),
                broker_hosts=',\n'.join(['"{}"'.format(host) for host in CREDIT_BROCKERS_HOSTS]),
                broker_urls=',\n'.join(['"{}"'.format(url) for url in CREDIT_BROCKERS_URLS]),
                betting_hosts=',\n'.join(['"{}"'.format(host) for host in BETTING_HOSTS]),
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


class Proleads(RegularSegmentBuilder):

    name_segment_dict = {
        'bank': 15781483,
        'mfo': 15781690,
        'broker': 15781687,
        'betting': 15781693,
    }

    keyword = 557
    number_of_days = 35

    def requires(self):
        return LogProcessor(ProcessedMetricsForProleads, self.date, self.number_of_days)

    def build_segment(self, inputs, output_path):
        self.yql.query(
            build_segment_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
