#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.lib.python import templater
from crypta.profile.utils import luigi_utils
from crypta.profile.utils.segment_utils import (
    builders,
    processors,
)

REALTY_EVENTS = '//home/verticals/broker/prod/warehouse/realty/event/1d'
REALTY_SITES = '//home/verticals/realty/production/export/resources/sites'

REALTY_QUERY_TEMPLATE = u"""
$realty_sites = (
    SELECT
        site_id,
        CASE
            WHEN site.building_class_int = 0 THEN 'UNKNOWN'
            WHEN site.building_class_int = 1 THEN 'ECONOM'
            WHEN site.building_class_int = 2 THEN 'BUSINESS'
            WHEN site.building_class_int = 3 THEN 'ELITE'
            WHEN site.building_class_int = 4 THEN 'COMFORT'
            WHEN site.building_class_int = 5 THEN 'COMFORT_PLUS'
            ELSE 'UNKNOWN'
        END AS building_class,
    FROM
      `{{realty_sites}}`
);

$realty_events_log = (
    SELECT
        object_info.offerInfo.offer_type_field AS offer_type,
        object_info.offerInfo.offer_category_field AS offer_category,
        CAST(object_info.site_info.site_id as Int64) AS site_id,
        event_type,
        user_info.web_user_info.user_yandex_uid AS yuid,
        user_info.app_user_info.mobile_uuid AS mobile_uuid,
    FROM `{{realty_events}}`
);

$make_id_pack = ($id, $id_type) -> {
    RETURN AsStruct($id AS id, $id_type AS id_type);
};

$stats_sites = (
    SELECT
      CASE
        WHEN mobile_uuid IS NOT NULL THEN $make_id_pack(mobile_uuid, 'uuid')
        WHEN yuid IS NOT NULL THEN $make_id_pack(yuid, 'yandexuid')
        ELSE NULL
      END AS id_pack,
      building_class
    FROM
      $realty_events_log as l
      LEFT JOIN $realty_sites AS s
      ON s.site_id == l.site_id
    WHERE
      event_type IN ('PHONE_SHOW', 'PHONE_CALL')
      AND l.offer_type == 'SELL'
      AND l.offer_category == 'APARTMENT'
      and l.site_id IS NOT NULL
);

INSERT INTO `{{output_table}}` WITH TRUNCATE
SELECT
    CASE
        WHEN building_class IN ('BUSINESS', 'ELITE') THEN 'realty_expensive'
        ELSE 'realty_not_expensive'
    END AS segment_name,
    id_pack.id AS id,
    id_pack.id_type AS id_type
FROM $stats_sites
WHERE
    building_class IN ('BUSINESS', 'ELITE', 'COMFORT', 'COMFORT_PLUS', 'ECONOM')
    AND id_pack IS NOT NULL
    AND id_pack.id != "";
"""


class ProcessedRealtyLogForRealtyInteractions(processors.DayProcessor):
    def requires(self):
        return {
            'realty_events': luigi_utils.ExternalInput(os.path.join(REALTY_EVENTS, self.date)),
            'realty_sites': luigi_utils.ExternalInput(REALTY_SITES),
        }

    def process_day(self, inputs, output_path):
        query = templater.render_template(
            REALTY_QUERY_TEMPLATE,
            vars={
                'realty_events': inputs['realty_events'].table,
                'realty_sites': inputs['realty_sites'].table,
                'output_table': output_path,
            },
            strict=True,
        )
        self.yql.query(
            query,
            transaction=self.transaction,
        )


class RealtyInteractions(builders.RegularSegmentBuilder):
    number_of_days = 35

    name_segment_dict = {
        'realty_not_expensive': (557, 19790530),
        'realty_expensive': (557, 19790528),
    }

    def requires(self):
        return processors.LogProcessor(ProcessedRealtyLogForRealtyInteractions, self.date, self.number_of_days)

    def build_segment(self, inputs, output_path):

        self.yt.unique(
            inputs.table,
            output_path,
            unique_by=('id', 'id_type', 'segment_name'),
        )
