#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import AttributeExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor

from crypta.profile.runners.segments.lib.coded_segments.data.seo_hosts import SEO_HOSTS

day_processor_query = """
$hosts = AsSet(
{hosts}
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT DISTINCT yandexuid, host
FROM `{input_table}`
WHERE DictContains($hosts, host) AND yandexuid IS NOT NULL AND yandexuid != 0
"""


class ProcessedMetricsForSeoSpecialists(DayProcessor):
    def requires(self):
        return AttributeExternalInput(
            os.path.join(
                config.METRICS_PARSED_DIR,
                self.date,
            ),
            attribute_name='closed',
            attribute_value=True,
        )

    def process_day(self, inputs, output_path):
        self.yql.query(
            day_processor_query.format(
                hosts='\n'.join("'{}',".format(host) for host in SEO_HOSTS),
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


class ProcessedBarForSeoSpecialists(DayProcessor):
    def requires(self):
        return AttributeExternalInput(
            os.path.join(
                config.BAR_PARSED_DIR,
                self.date,
            ),
            attribute_name='closed',
            attribute_value=True,
        )

    def process_day(self, inputs, output_path):
        self.yql.query(
            day_processor_query.format(
                hosts='\n'.join("'{}',".format(host) for host in SEO_HOSTS),
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


segment_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    CAST(yandexuid AS String) AS id,
    'yandexuid' AS id_type,
    'seo_specialists' AS segment_name
FROM (
    SELECT yandexuid, host
    FROM `{metrics_table}`
    UNION ALL
    SELECT yandexuid, host
    FROM `{bar_table}`
)
GROUP BY yandexuid
HAVING ListLength(AGGREGATE_LIST_DISTINCT(host)) > 1
"""


class SeoSpecialists(RegularSegmentBuilder):
    name_segment_dict = {
        'seo_specialists': (549, 2027),
    }
    number_of_days = 35

    def requires(self):
        return {
            'metrics': LogProcessor(
                ProcessedMetricsForSeoSpecialists,
                self.date,
                self.number_of_days,
            ),
            'bar': LogProcessor(
                ProcessedBarForSeoSpecialists,
                self.date,
                self.number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            segment_query.format(
                metrics_table=inputs['metrics'].table,
                bar_table=inputs['bar'].table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
