#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput, AttributeExternalInput, YtTarget
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


edadeal_day_processor_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    `uuid`,
    `id`,
    "{date}" AS `date`,
FROM `{input_table}`
WHERE `uuid` IS NOT NULL
    AND updated_at >= {min_ts}
    AND updated_at < {max_ts}
"""


class ProcessedEdadealReceiptsForSmokers(DayProcessor):
    def requires(self):
        return ExternalInput(os.path.join(config.EDADEAL_RECEIPTS, self.date))

    def process_day(self, inputs, output_path):
        self.yql.query(
            edadeal_day_processor_query.format(
                input_table=inputs.table,
                output_table=output_path,
                min_ts=date_helpers.from_date_string_to_timestamp(self.date),
                max_ts=date_helpers.from_date_string_to_timestamp(date_helpers.get_date_from_future(self.date, days=1)),
                date=self.date,
            ),
            transaction=self.transaction,
        )

    def output(self):
        return YtTarget(self._day_processor_output, allow_empty=True)


geocube_day_processor_query = u"""
$smoker_lemmas = ToSet(AsList('iqos', 'glo', 'табак', 'сигарета', 'juul', 'кальян', 'кальянная'));

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    IF(String::StartsWith(id, 'y'), SUBSTRING(id, 1), id) AS id,
    id_type,
    'smokers' AS segment_name
FROM `{input_table}`
WHERE DictLength(SetIntersection(ToSet(lemmas), $smoker_lemmas)) > 0
"""


class ProcessedGeocubeForSmokers(DayProcessor):
    def requires(self):
        return AttributeExternalInput(
            os.path.join(config.GEOCUBE_PARSED_DIR, self.date),
            attribute_name='closed',
            attribute_value=True,
        )

    def process_day(self, inputs, output_path):
        self.yql.query(
            geocube_day_processor_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


segment_query = u"""
$receipts_filtered_by_attrs = (
    SELECT
        events.`uuid` AS id,
        'uuid' AS id_type,
        events.`date` AS event_date,
        'smokers' AS segment_name
    FROM `{edadeal_events_table}` AS events
    INNER JOIN `{attributes_table}` AS attributes
    ON events.`id` == attributes.item_id
    WHERE attributes.attr_482 IS NOT NULL
        AND CAST(attributes.attr_482 AS String) IN ('Сигареты', 'Зажигалка')
);

$receipts = (
    SELECT id, id_type, segment_name
    FROM $receipts_filtered_by_attrs
    GROUP BY id, id_type, segment_name
    HAVING COUNT(DISTINCT event_date) > 1
);

INSERT INTO `{output_path}` WITH TRUNCATE
SELECT id, id_type, segment_name
FROM $receipts
GROUP BY id, id_type, segment_name

UNION ALL

SELECT id, id_type, segment_name
FROM `{geocube_table}`
GROUP BY id, id_type, segment_name
"""


class Smokers(RegularSegmentBuilder):
    name_segment_dict = {
        'smokers': (557, 11682883),
    }

    number_of_days = 35
    edadeal_number_of_days = 90

    def requires(self):
        return {
            'ProcessedGeocube': LogProcessor(
                ProcessedGeocubeForSmokers,
                self.date,
                self.number_of_days,
            ),
            'ProcessedEdadealReceipts': LogProcessor(
                ProcessedEdadealReceiptsForSmokers,
                self.date,
                self.edadeal_number_of_days,
            ),
            'edadeal_item_attributes': ExternalInput(os.path.join(config.EDADEAL_ITEM_ATTRIBUTES, '0000-00-00')),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            query_string=segment_query.format(
                edadeal_events_table=inputs['ProcessedEdadealReceipts'].table,
                attributes_table=inputs['edadeal_item_attributes'].table,
                edadeal_start_date=date_helpers.get_date_from_past(self.date, self.edadeal_number_of_days),
                geocube_table=inputs['ProcessedGeocube'].table,
                output_path=output_path,
            ),
            transaction=self.transaction,
        )
