#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor

day_processor_query_template = u"""
$live_in_moscow_and_piter = (
    SELECT
        mmetric_devid,
        CASE
            WHEN city_geoid == 213 THEN 'moscow'
            ELSE 'piter'
        END as home
    FROM `{homes}`
    WHERE city_geoid == 213 OR city_geoid == 2 --может быть стоит быть строже, выделить районы
);

$region_visits = (
    SELECT
        mmetric_devid,
        region_id,
    FROM `{geo_table}`
    WHERE geo.source IN ("GT_GPS", "GT_LBS", "GT_PLATFORM_LBS") AND geo.precision < 500
    GROUP BY
        mm_device_id AS mmetric_devid,
        Geo::RegionByLocation(geo.lat, geo.lon).id AS region_id
);

$live_in_moscow_and_piter_visits = (
    SELECT
        rv.mmetric_devid AS mmetric_devid,
        Geo::RoundRegionById(region_id, "city").id == 213 AS is_moscow,
        Geo::RoundRegionById(region_id, "region").id == 1 AS is_moscow_oblast,
        Geo::RoundRegionById(region_id, "city").id == 2 AS is_piter,
        Geo::RoundRegionById(region_id, "region").id == 10174 AS is_lenin_oblast,
        home
    FROM $live_in_moscow_and_piter AS lm
    INNER JOIN $region_visits AS rv
    USING(mmetric_devid)
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    mmetric_devid,
    '{date}' AS `date`,
    CASE
        WHEN NOT MAX(is_moscow) AND MAX(is_moscow_oblast) AND home == 'moscow' THEN True
        WHEN NOT MAX(is_piter) AND MAX(is_lenin_oblast) AND home == 'piter' THEN True
        ELSE False
    END AS in_dacha,
    home
FROM $live_in_moscow_and_piter_visits
GROUP BY mmetric_devid, home;
"""

summer_residents_query_template = """
$dacha_visits_count = (
    SELECT
        mmetric_devid,
        home,
        COUNT_IF('{start_date_this_week}' <= `date` AND '{end_date_this_week}' >= `date` AND in_dacha) AS this_week,
        COUNT_IF('{start_date_previous_week}' <= `date` AND '{end_date_previous_week}' >= `date` AND in_dacha) AS previous_week
    FROM `{input_table}`
    GROUP BY mmetric_devid, home
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    mmetric_devid as id,
    'mm_device_id' as id_type,
    home AS segment_name
FROM $dacha_visits_count
WHERE this_week >= 2 AND previous_week >= 2;
"""


class ProcessedGeoLogForSummerResidents(DayProcessor):
    def requires(self):
        return {
            'OrgVisitsGeo': ExternalInput(os.path.join(config.ORG_VISITS_GEO, self.date)),
            'UserHomes': ExternalInput(config.USER_HOMES),
        }

    def process_day(self, inputs, output_path):
        self.yql.query(
            day_processor_query_template.format(
                homes=inputs['UserHomes'].table,
                geo_table=inputs['OrgVisitsGeo'].table,
                output_table=output_path,
                date=self.date,
            ),
            transaction=self.transaction,
        )


class SummerResidents(RegularSegmentBuilder):
    keyword = 557
    number_of_days = 14

    name_segment_dict = {
        'moscow': 16207957,
        'piter': 16207882,
    }

    def requires(self):
        return LogProcessor(ProcessedGeoLogForSummerResidents, self.date, self.number_of_days)

    def build_segment(self, inputs, output_path):

        self.yql.query(
            summer_residents_query_template.format(
                input_table=inputs.table,
                output_table=output_path,
                start_date_this_week=date_helpers.get_date_from_past(self.date, 6),
                end_date_this_week=self.date,
                start_date_previous_week=date_helpers.get_date_from_past(self.date, 13),
                end_date_previous_week=date_helpers.get_date_from_past(self.date, 7),
            ),
            transaction=self.transaction,
        )
