#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.utils.config import config
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor
from crypta.profile.utils.luigi_utils import ExternalInput, OldNodesByNameCleaner

travellers_weekly_query_template = """
PRAGMA yt.ForceInferSchema = '1';

$has_gps = ($x) -> {{
    RETURN Yson::ConvertToInt64($x["accuracy"][0][1]) != 0;
}};

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    id,
    id_type,
    Geo::RegionByLocation(Yson::ConvertToDouble(res['center']['lat']), Yson::ConvertToDouble(res['center']['lon'])).id as region,
    DictKeys(Yson::ConvertToDict(res['days_radiance'])) AS days,
    week
FROM(
    SELECT
        unified_id AS id,
        source_unified_id AS id_type,
        Yson::ConvertToList(`result`["cluster_list"]) AS res,
        TableName() AS week
    FROM `{weekly_geo}`
    )
FLATTEN LIST BY res
WHERE $has_gps(res)
"""

travellers_query_template = """
$parse = DateTime::Parse("%Y-%m-%d");

$travell_matched = (
SELECT
    cryptaId AS crypta_id,
    Geo::RoundRegionById(visits.region, "region").id AS region,
    days AS dates,
    week AS week_end_date,
FROM CONCAT({region_visits}) AS visits
INNER JOIN `{vertices_no_multi_profile}` AS matching
USING(id, id_type)

UNION ALL

SELECT
    id AS crypta_id,
    Geo::RoundRegionById(visits.region, "region").id AS region,
    days AS dates,
    week AS week_end_date,
FROM CONCAT({region_visits}) AS visits
WHERE id_type == 'crypta_id'
);

$travell_visits = (
SELECT
    region,
    dates AS `date`,
    crypta_id,
    week_end_date,
    main_region
FROM (
    SELECT
        home.crypta_id AS crypta_id,
        visits.region AS region,
        dates,
        week_end_date,
        home.main_region as main_region
    FROM `{crypta_id_regions}` AS home
    INNER JOIN $travell_matched AS visits
    USING(crypta_id)
    WHERE visits.region NOT IN AsSet(home.main_region, -1)
    )
FLATTEN LIST BY dates
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    id,
    id_type,
    segment_name
FROM(
    SELECT
        crypta_id AS id,
        'crypta_id' AS id_type,
        CASE
            WHEN Geo::RoundRegionById(region, "country").id != Geo::RoundRegionById(CAST(main_region AS Int32), "country").id THEN 'international'
            ELSE 'domestic'
        END AS segment_name,
        MAX(`date`) AS last_seen,
        MIN(`date`) AS first_seen,
        region,
        week_end_date,
    FROM $travell_visits
    GROUP BY region, main_region, crypta_id, week_end_date
)
WHERE
    last_seen <= week_end_date AND
    DateTime::ToDays(DateTime::MakeTimestamp($parse(last_seen)) - DateTime::MakeTimestamp($parse(first_seen))) > 0 AND
    DateTime::ToDays(DateTime::MakeTimestamp($parse(week_end_date)) - DateTime::MakeTimestamp($parse(first_seen))) <= 7
GROUP BY id, id_type, segment_name
"""


class TravellersWeekly(DayProcessor):
    def requires(self):
        return {
            'WeeklyGeoTable': ExternalInput(os.path.join(config.WEEKLY_GEO_FOLDER, self.date)),
        }

    def process_day(self, inputs, output_path):
        self.yql.query(
            travellers_weekly_query_template.format(
                weekly_geo=inputs['WeeklyGeoTable'].table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


class Travellers(RegularSegmentBuilder):
    keyword = 547
    name_segment_dict = {
        'international': 1192,
        'domestic': 1193,
    }

    def requires(self):
        dates = sorted(self.yt.list(config.WEEKLY_GEO_FOLDER))[-52:]

        tasks_to_run = []
        for date in dates:
            tasks_to_run.append(TravellersWeekly(date))

        return {
            'DayProcessorTasks': tasks_to_run,
            'CleanOldDayProcessors': OldNodesByNameCleaner(
                date=self.date,
                folder=tasks_to_run[0].output_directory,
                lifetime=372,
                date_format='%Y-%m-%d',
            ),
            'CryptaIdRegions': ExternalInput(config.GEO_CRYPTA_ID),
        }

    def build_segment(self, inputs, output_path):
        last_processed = self.yt.TablePath(os.path.join(config.WEEKLY_GEO_FOLDER, self.yt.list(config.WEEKLY_GEO_FOLDER)[-1]))

        if self.yt.get_attribute(last_processed, 'modification_time').split('T')[0] != self.date:
            self.yt.set_attribute(
                output_path,
                'generate_date',
                self.date,
            )
            return

        input_tables = []
        for target in self.input()['DayProcessorTasks']:
            input_tables.append("`{}`".format(target.table))

        with self.yt.Transaction():
            self.yql.query(
                travellers_query_template.format(
                    region_visits=', '.join(input_tables),
                    vertices_no_multi_profile=config.VERTICES_NO_MULTI_PROFILE,
                    crypta_id_regions=inputs['CryptaIdRegions'].table,
                    output_table=output_path,
                ),
                transaction=self.transaction,
            )

            self.yt.set_attribute(
                output_path,
                'generate_date',
                self.date,
            )
