#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import AttributeExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


day_processor_query = u"""
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT id, id_type, segment_name, host
FROM (
    SELECT
        CAST(yandexuid AS String) AS id,
        'yandexuid' AS id_type,
        'want_to_change_the_provider' AS segment_name,
        String::ToLower(host) AS host
    FROM `{input_table}`
    WHERE ListHas(Yson::ConvertToStringList(lemmas), 'тариф')
        AND (ListHas(Yson::ConvertToStringList(lemmas), 'провайдер') OR ListHas(Yson::ConvertToStringList(lemmas), 'интернет'))
        AND host IS NOT NULL
        AND host != ''
        AND yandexuid != 0
)
GROUP BY id, id_type, segment_name, host
"""


class ProcessedMetricsForWantToChangeTheProvider(DayProcessor):
    def requires(self):
        return AttributeExternalInput(
            os.path.join(
                config.METRICS_PARSED_DIR,
                self.date,
            ),
            attribute_name='closed',
            attribute_value=True,
        )

    def process_day(self, inputs, output_path):
        self.yql.query(
            day_processor_query.format(
                input_table=inputs.table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )


segment_query = """
INSERT INTO `{output_table}` WITH TRUNCATE
SELECT id, id_type, segment_name
FROM `{input_table}`
GROUP BY id, id_type, segment_name
HAVING ListLength(AGGREGATE_LIST_DISTINCT(host)) >= 3
    AND ListLength(AGGREGATE_LIST_DISTINCT(host)) <= 25
"""


class WantToChangeTheProvider(RegularSegmentBuilder):
    name_segment_dict = {
        'want_to_change_the_provider': (557, 17984218),
    }

    number_of_days = 35

    def requires(self):
        return {
            'ProcessedMetrics': LogProcessor(
                ProcessedMetricsForWantToChangeTheProvider,
                self.date,
                self.number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            segment_query.format(
                input_table=inputs['ProcessedMetrics'].table,
                output_table=output_path,
            ),
            transaction=self.transaction,
        )
