#!/usr/bin/env python
# -*- coding: utf-8 -*-
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.config import config


turbo_users_query = """
PRAGMA yt.InferSchema = '1';

$all_webmaster_users = (
    SELECT
        domain,
        user_id,
    FROM (
        SELECT
            user_id,
            Url::CutWWW(Url::GetHost(Url::HostNameToPunycode(Url::NormalizeWithDefaultHttpScheme(host_url)))) AS domain
        FROM `{webmaster_verified_hosts_table}`
    )
    GROUP BY domain, user_id
);

$all_webmaster_users_with_metrica_counter = (
    SELECT
        all_users.user_id as user_id,
        CASE
            WHEN metrica_counter.metrica_visitors_count IS NOT NULL THEN metrica_counter.metrica_visitors_count
            ELSE metrica_counter.bar_visitors_count
        END as visitors_count
    FROM $all_webmaster_users as all_users
    INNER JOIN `{metrica_counter}` as metrica_counter
    ON metrica_counter.site == all_users.domain
    WHERE metrica_counter.bar_visitors_count IS NOT NULL OR metrica_counter.metrica_visitors_count IS NOT NULL
);

$all_webmaster_users_with_metrica_counter_and_timestamp = (
    SELECT
        all_with_counter.user_id as user_id,
        all_with_counter.visitors_count as visitors_count,
        CAST(last_visit.last_visit as Uint64) as last_visit
    FROM $all_webmaster_users_with_metrica_counter as all_with_counter
    INNER JOIN `{last_visit}` as last_visit
    USING(user_id)
);

$result_without_timestamp = (
    SELECT
         user_id as id,
         'puid' as id_type,
         last_visit,
         CASE
            WHEN visitors_count < 500000 THEN 'small_all'
            WHEN visitors_count >= 500000 AND visitors_count < 1000000 THEN 'medium_all'
            WHEN visitors_count >= 1000000 THEN 'big_all'
            ELSE ''
        END AS segment_name,
    FROM $all_webmaster_users_with_metrica_counter_and_timestamp
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    id,
    id_type,
    segment_name
FROM $result_without_timestamp

UNION ALL

SELECT
    user_id as id,
    'puid' as id_type,
    'webmaster_users' AS segment_name
FROM $all_webmaster_users

UNION ALL

SELECT
    user_id as id,
    'puid' as id_type,
    'turbo_users' AS segment_name
FROM $all_webmaster_users
WHERE domain IN (
    SELECT Url::GetHost(Url::HostNameToPunycode(Url::NormalizeWithDefaultHttpScheme(Host))) AS domain
    FROM `{turbo_hosts_table}`
);
"""


class Webmaster(RegularSegmentBuilder):
    name_segment_dict = {
        'small_all': (557, 13247368),
        'medium_all': (557, 13247380),
        'big_all': (557, 13247398),
        'turbo_users': (549, 1214),
        'webmaster_users': (549, 1217),
    }

    def requires(self):
        return {
            'webmaster_verified_hosts': ExternalInput(
                config.WEBMASTER_VERIFIED_HOSTS_TABLE_TEMPLATE.format(self.date.replace('-', ''))
            ),
            'metrica_counter': ExternalInput(config.YANDEXUID_METRICA_BROWSER_COUNTER_TABLE),
            'last_visit': ExternalInput(config.WEBMASTER_LAST_VISIT_TABLE),
            'turbo_hosts': ExternalInput(config.TURBO_HOSTS_TABLE),
        }

    def build_segment(self, inputs, output_path):
        self.yql.query(
            turbo_users_query.format(
                webmaster_verified_hosts_table=inputs['webmaster_verified_hosts'].table,
                metrica_counter=inputs['metrica_counter'].table,
                output_table=output_path,
                last_visit=inputs['last_visit'].table,
                turbo_hosts_table=inputs['turbo_hosts'].table,
            ),
            transaction=self.transaction,
        )
