#!/usr/bin/env python
# -*- coding: utf-8 -*-

from datetime import datetime

from crypta.profile.utils.segment_utils.builders import RegularSegmentBuilder
from crypta.profile.utils.clients.clickhouse import ClickhouseClient
from crypta.profile.utils.segment_utils.processors import DayProcessor, LogProcessor


daily_template = """
SELECT DISTINCT
    id,
    widget,
    source
FROM (
    SELECT
        DeviceID AS id,
        if(
            EventName == 'widget_installations'
            AND (
                ParsedParams.Key2[indexOf(ParsedParams.Key1, 'installations')] like '%search_medium%'
                OR ParsedParams.Key2[indexOf(ParsedParams.Key1, 'installations')] like '%weather_medium%'
                OR ParsedParams.Key2[indexOf(ParsedParams.Key1, 'installations')] like '%weather_small%'
            ),
            1,
            0
        ) as widget,
        'bro' AS source
    FROM mobgiga.events_all
    WHERE APIKey == 19531
        AND EventDate == '{date}'
        AND OSVersion like '14%'
        AND AppVersionName >= '20.8.2.298'


    UNION ALL

    SELECT
        DeviceID AS id,
        if(
            EventName == 'widget_installations'
            AND (
                ParsedParams.Key2[indexOf(ParsedParams.Key1, 'installations')] like '%search_medium%'
                OR ParsedParams.Key2[indexOf(ParsedParams.Key1, 'installations')] like '%weather_medium%'
                OR ParsedParams.Key2[indexOf(ParsedParams.Key1, 'installations')] like '%weather_small%'
            ),
            1,
            0
        ) as widget,
        'pp' AS source
    FROM mobgiga.events_all
    WHERE APIKey == 42989
        AND EventDate == '{date}'
        AND OSVersion like '14%'
        AND (AppVersionName like '4000%')
);
"""


def segment_name_reducer(key, rows):
    widget = False

    for row in rows:
        if 'widget' not in row:
            return

        if row['widget'] == '1':
            widget = True
            break

    if widget:
        yield {
            'id': key['id'],
            'id_type': 'mm_device_id',
            'segment_name': key['source'] + '_widget',
        }

    yield {
        'id': key['id'],
        'id_type': 'mm_device_id',
        'segment_name': key['source'],
    }


class ProcessedEventsAllForWidgets(DayProcessor):
    def requires(self):
        return []

    def process_day(self, inputs, output_path):
        clickhouse = ClickhouseClient(logger=self.logger)
        result = clickhouse.make_query(
            daily_template.format(date=self.date)
        )
        if not result:
            self.logger.error('Empty result')
        entries = clickhouse.result_to_dict_list(result, ['id', 'widget', 'source'])

        self.yt.write_table(output_path, entries)


class Widgets(RegularSegmentBuilder):
    name_segment_dict = {
        'pp': 17448481,
        'bro': 17448475,
        'pp_widget': 17448484,
        'bro_widget': 17448478,
    }

    keyword = 557

    def requires(self):
        number_of_days = (datetime.strptime(self.date, "%Y-%m-%d").date() - datetime.strptime('2020-09-16', "%Y-%m-%d").date()).days

        return {
            'ProcessedMetrics': LogProcessor(
                ProcessedEventsAllForWidgets,
                self.date,
                number_of_days,
            ),
        }

    def build_segment(self, inputs, output_path):
        self.yt.run_map_reduce(
            None,
            segment_name_reducer,
            inputs['ProcessedMetrics'].table,
            output_path,
            reduce_by=['id', 'source'],
        )
