import collections
import os

import luigi

from crypta.profile.lib.frozen_dict import FrozenDict
from crypta.profile.lib.segments import condition_helpers
from crypta.profile.runners.segments.lib.constructor_segments.common import utils
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import AttributeExternalInput


yql_request = """
$catalogia_id_to_rule_revision_id = AsDict(
    {catalogia_id_to_rule_revision_id}
);

$format_to_date = DateTime::Format("%Y-%m-%d");

$yandexuid_with_rule_ids = (
    SELECT
        yandexuid,
        $catalogia_id_to_rule_revision_id[category] AS rule_ids,
        `date`,
    FROM `{source_table}`
    WHERE DictContains($catalogia_id_to_rule_revision_id, category)
    GROUP BY yandexuid, category, $format_to_date(CAST(`timestamp` AS Datetime)) AS `date`
);

INSERT INTO `{destination_table}` WITH TRUNCATE
SELECT DISTINCT
    yandexuid,
    rule_id,
    `date`,
FROM $yandexuid_with_rule_ids
FLATTEN LIST BY rule_ids AS rule_id;
"""


class GetStandardSegmentsByCatalogiaDailyProcessor(utils.DailyRulesProcessor):
    # https://catmedia.yandex.ru/acgi-bin/interface/get_categs_tree.pl
    task_group = 'constructor_segments'
    catalogia_id_to_rule_revision_id = luigi.Parameter(significant=False)
    priority = 100

    def __init__(self, *args, **kwargs):
        super(GetStandardSegmentsByCatalogiaDailyProcessor, self).__init__(*args, **kwargs)

    def requires(self):
        return AttributeExternalInput(
            os.path.join(
                config.BB_PARSED_DIR,
                self.date,
            ),
            attribute_name='closed',
            attribute_value=True,
        )

    def compute(self, input_table, output_table, tx):
        catalogia_id_to_rule_revision_id = dict(self.catalogia_id_to_rule_revision_id)
        self.logger.info('catalogia_id to rule revision ids: {}'.format(catalogia_id_to_rule_revision_id))

        self.yt.create_empty_table(
            output_table,
            schema=utils.daily_schema,
        )

        tuple_strings = []

        for catalogia_id, rule_revision_ids in catalogia_id_to_rule_revision_id.iteritems():
            rule_revision_id_strings = []
            for rule_revision_id in rule_revision_ids:
                if not self.rule_revision_ids_to_be_prepared or rule_revision_id in self.rule_revision_ids_to_be_prepared:
                    rule_revision_id_strings.append('{}ul'.format(rule_revision_id))
            if rule_revision_id_strings:
                tuple_strings.append('({}ul, AsList({})),'.format(catalogia_id, ', '.join(rule_revision_id_strings)))

        if tuple_strings:
            self.yql.query(
                yql_request.format(
                    source_table=input_table,
                    destination_table=output_table,
                    catalogia_id_to_rule_revision_id='\n'.join(tuple_strings),
                ),
                transaction=tx,
            )

    @classmethod
    def prepare_rules(cls, rule_conditions, segments_config):
        values_to_rule_revision_id = collections.defaultdict(set)

        for rule_condition in rule_conditions:
            for value in condition_helpers.normalize_categories(rule_condition.values):
                values_to_rule_revision_id[value].add(rule_condition.revision)

        return {"catalogia_id_to_rule_revision_id": FrozenDict(values_to_rule_revision_id)}, []
