from __future__ import unicode_literals
import collections
import os

import luigi

from crypta.lab.proto import constructor_pb2
from crypta.profile.lib import date_helpers
from crypta.profile.lib.frozen_dict import FrozenDict
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInput
from crypta.profile.runners.segments.lib.constructor_segments.common.utils import DailyRulesProcessor


efir_query = u"""
$channel_to_rule_revision_id_dict = AsDict(
{channel_to_rule_revision_id}
);

$program_to_rule_revision_id_dict = AsDict(
{program_to_rule_revision_id}
);

$channel_patterns = DictKeys($channel_to_rule_revision_id_dict);
$program_patterns = DictKeys($program_to_rule_revision_id_dict);

$add_point = ($pattern) -> {{RETURN '.*'||$pattern||'.*'}};

$match = Hyperscan::MultiMatch(ListConcat(ListMap($channel_patterns, ($pattern) -> {{RETURN $add_point($pattern)}}), "
") ?? "");

$convert_condition_results_to_names = ($pattern_results, $patterns) -> {{
    RETURN ListMap(
        ListEnumerate($patterns),
        ($patterns_and_ids) -> {{
            RETURN CASE $pattern_results[$patterns_and_ids.0]
                    WHEN True THEN $patterns_and_ids.1
                    ELSE NULL
                    END;
        }}
    )
}};

$channel_ids_with_view_time = (
SELECT *
FROM (
    SELECT
        yandexuid,
        ListFlatMap(
            ListFlatMap(names, ($x) -> {{RETURN $channel_to_rule_revision_id_dict[$x];}}),
            ($x) -> {{RETURN $x;}}
        ) AS rule_revision_id,
        view_time
    FROM (
        SELECT
            yandexuid,
            $convert_condition_results_to_names(YQL::ToList($match(channel)), $channel_patterns) AS names,
            view_time
        FROM `{input_table}`
        WHERE CAST(yandexuid AS Uint64) IS NOT NULL AND CAST(yandexuid AS Uint64) != 0
    )
)
WHERE ListLength(rule_revision_id) > 0
);

$program_ids_with_view_time = (
SELECT *
FROM (
    SELECT
        yandexuid,
        ListFlatMap(
            ListFlatMap(names, ($x) -> {{RETURN $program_to_rule_revision_id_dict[$x];}}),
            ($x) -> {{RETURN $x;}}
        ) AS rule_revision_id,
        view_time
    FROM (
        SELECT
            yandexuid,
            $convert_condition_results_to_names(YQL::ToList($match(program)), $program_patterns) AS names,
            view_time
        FROM `{input_table}`
        WHERE CAST(yandexuid AS Uint64) IS NOT NULL AND CAST(yandexuid AS Uint64) != 0
    )
)
WHERE ListLength(rule_revision_id) > 0
);

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    CAST(yandexuid AS Uint64) AS yandexuid,
    rule_revision_id AS rule_id
FROM (
    SELECT *
    FROM (
        SELECT
            yandexuid,
            rule_revision_id,
            view_time
        FROM $channel_ids_with_view_time

        UNION ALL

        SELECT
            yandexuid,
            rule_revision_id,
            view_time
        FROM $program_ids_with_view_time
    )
    FLATTEN LIST BY rule_revision_id
)
GROUP BY rule_revision_id, yandexuid
HAVING SUM(view_time) > 30
"""


class GetStandardSegmentsByEfirData(DailyRulesProcessor):
    task_group = 'constructor_segments'
    efir_channels_to_rule_revision_id = luigi.Parameter(significant=False)
    efir_programs_to_rule_revision_id = luigi.Parameter(significant=False)

    priority = 100

    def requires(self):
        return ExternalInput(
            table=os.path.join(config.VIDEO_SESSIONS_LOG_DIRECTORY, date_helpers.get_yesterday(self.date), 'sessions'),
        )

    def prepare_content(self, content_type, efir_content_to_rule_revision_id):
        efir_content_to_rule_revision_id = dict(efir_content_to_rule_revision_id)
        self.logger.info('efir {} to rule revision ids: {}'.format(content_type, efir_content_to_rule_revision_id))

        tuple_strings = []

        for efir_content, rule_revision_id_set in efir_content_to_rule_revision_id.iteritems():
            rule_revision_id_strings = []
            for rule_revision_id in rule_revision_id_set:
                if not self.rule_revision_ids_to_be_prepared or \
                        rule_revision_id in self.rule_revision_ids_to_be_prepared:
                    rule_revision_id_strings.append(u'{}ul'.format(rule_revision_id))

            if rule_revision_id_strings:
                tuple_strings.append(u'("{}", AsList({})),'.format(efir_content.replace('"', '\\"'), ', '.join(rule_revision_id_strings)))

        return tuple_strings

    def compute(self, input_table, output_table, tx):
        channel_strings = self.prepare_content('channels', self.efir_channels_to_rule_revision_id)
        program_strings = self.prepare_content('programs', self.efir_programs_to_rule_revision_id)

        self.yql.query(
            efir_query.format(
                input_table=input_table,
                output_table=output_table,
                channel_to_rule_revision_id=u'\n'.join(channel_strings),
                program_to_rule_revision_id=u'\n'.join(program_strings),
            ),
            transaction=tx,
        )

    @classmethod
    def prepare_rules(cls, rule_conditions, segments_config):
        efir_channels_to_rule_revision_id = collections.defaultdict(set)
        efir_programs_to_rule_revision_id = collections.defaultdict(set)

        for rule_condition in rule_conditions:
            if rule_condition.source == constructor_pb2.RuleCondition.Source.Name(constructor_pb2.RuleCondition.Source.EFIR_CHANNELS):
                for channel in rule_condition.values:
                    efir_channels_to_rule_revision_id[channel.strip()].add(rule_condition.revision)

            elif rule_condition.source == constructor_pb2.RuleCondition.Source.Name(constructor_pb2.RuleCondition.Source.EFIR_PROGRAMS):
                for program in rule_condition.values:
                    efir_programs_to_rule_revision_id[program.strip()].add(rule_condition.revision)

        return {
            "efir_channels_to_rule_revision_id": FrozenDict(efir_channels_to_rule_revision_id),
            "efir_programs_to_rule_revision_id": FrozenDict(efir_programs_to_rule_revision_id),
        }, []
