import collections
import os

import luigi

from crypta.lib.python import templater
from crypta.profile.lib.frozen_dict import FrozenDict
from crypta.profile.lib.segments import condition_helpers
from crypta.profile.runners.segments.lib.constructor_segments.common import utils
from crypta.profile.utils import (
    loggers,
    luigi_utils,
)
from crypta.profile.utils.config import config


class GetStandardSegmentsByMobileApp(luigi_utils.BaseYtTask):
    date = luigi.Parameter()
    task_group = 'constructor_segments'
    app_to_rule_lab_id = luigi.Parameter(significant=False)
    rule_revision_ids = luigi.Parameter(significant=False)

    priority = 100

    def requires(self):
        return luigi_utils.ExternalInput(config.DEVID_BY_APP_MONTHLY_TABLE)

    def run(self):
        self.logger.info('App to rule ids: {}'.format(dict(self.app_to_rule_lab_id)))
        with loggers.TimeTracker(self.__class__.__name__), self.yt.Transaction() as transaction:
            self.compute(
                self.output().table,
                transaction,
            )

            self.yt.set_attribute(self.output().table, 'generate_date', self.date)
            self.yt.set_attribute(self.output().table, 'rule_ids', sorted(self.rule_revision_ids))

    def compute(self, output_table, transaction):
        self.yt.create_empty_table(
            output_table,
            schema=utils.aggregated_schema,
        )

        if self.app_to_rule_lab_id:
            self.yql.query(
                query_string=templater.render_resource("/query/apps.yql", strict=True, vars={
                    "output_table": output_table,
                    "input_table": self.input().table,
                    "app_to_rule_lab_id": self.app_to_rule_lab_id,
                }),
                transaction=transaction,
            )

    def output(self):
        return luigi_utils.YtTableMultipleAttributeTarget(
            os.path.join(config.AGGREGATED_STANDARD_HEURISTIC_DIRECTORY, self.__class__.__name__),
            {
                'generate_date': self.date,
                'rule_ids': sorted(self.rule_revision_ids),
            }
        )

    @staticmethod
    def prepare_rules(rule_conditions, segments_config):
        app_to_rule_lab_id = collections.defaultdict(set)

        for rule_condition in rule_conditions:
            rule_lab_id = segments_config.rule_revision_id_to_rule_id[rule_condition.revision]
            normalized_apps = set(condition_helpers.normalize_apps(rule_condition.values))

            for app in normalized_apps.intersection(segments_config.existing_apps):
                app_to_rule_lab_id[app].add(rule_lab_id)

            non_existing_apps = normalized_apps.difference(segments_config.existing_apps)
            if non_existing_apps:
                segments_config.logger.info(u'Apps: {} do not exist'.format(u', '.join(non_existing_apps)))

        return {"app_to_rule_lab_id": FrozenDict(app_to_rule_lab_id)}, []
