import collections
import os

import luigi

from crypta.lib.python import templater
from crypta.profile.lib.frozen_dict import FrozenDict
from crypta.profile.lib.segments import condition_helpers
import crypta.profile.runners.segments.lib.constructor_segments.common.utils as standard_utils
from crypta.profile.utils import luigi_utils
from crypta.profile.utils.config import config


class GetStandardSegmentsByMusicLikes(luigi_utils.BaseYtTask):
    date = luigi.Parameter()
    task_group = 'constructor_segments'
    liked_artist_id_to_rule_lab_id = luigi.Parameter(significant=False)
    rule_revision_ids = luigi.Parameter(significant=False)

    priority = 100

    def requires(self):
        return {
            'like_events': luigi_utils.ExternalInput(
                os.path.join(
                    config.MUSIC_LIKE_EVENTS_LOG_FOLDER,
                    self.date,
                )
            ),
            'track_metadata': luigi_utils.ExternalInput(
                os.path.join(
                    config.MUSIC_TRACKS_METADATA,
                    self.date,
                )
            ),
            'album_metadata': luigi_utils.ExternalInput(
                os.path.join(
                    config.MUSIC_ALBUMS_METADATA,
                    self.date,
                )
            ),
        }

    def output(self):
        return luigi_utils.YtTableMultipleAttributeTarget(
            os.path.join(
                config.AGGREGATED_STANDARD_HEURISTIC_DIRECTORY,
                self.__class__.__name__,
            ),
            {
                'generate_date': self.date,
                'rule_ids': sorted(self.rule_revision_ids),
            },
        )

    def run(self):
        liked_artist_id_to_rule_lab_id = dict(self.liked_artist_id_to_rule_lab_id)
        self.logger.info('artist_id to rule lab ids: {}'.format(liked_artist_id_to_rule_lab_id))

        with self.yt.Transaction() as transaction:
            self.yt.create_empty_table(
                self.output().table,
                schema=standard_utils.aggregated_schema,
            )

            if liked_artist_id_to_rule_lab_id:
                self.yql.query(
                    query_string=templater.render_resource("/query/music.yql", strict=True, vars={
                        "output_table": self.output().table,
                        "like_events": self.input()['like_events'].table,
                        "artist_id_to_rule_lab_ids": liked_artist_id_to_rule_lab_id,
                        "track_metadata": self.input()['track_metadata'].table,
                        "album_metadata": self.input()['album_metadata'].table,
                    }),
                    transaction=transaction,
                )

            self.yt.set_attribute(self.output().table, 'generate_date', self.date)
            self.yt.set_attribute(self.output().table, 'rule_ids', sorted(self.rule_revision_ids))

    @classmethod
    def prepare_rules(cls, rule_conditions, segments_config):
        values_to_rule_revision_id = collections.defaultdict(set)

        for rule_condition in rule_conditions:
            for value in condition_helpers.normalize_artists(rule_condition.values):
                values_to_rule_revision_id[value].add(rule_condition.revision)

        return {"liked_artist_id_to_rule_lab_id": FrozenDict(values_to_rule_revision_id)}, []
