import collections
import json
import os

import luigi
from yt.common import YtResponseError
import yt.wrapper as yt

from crypta.lib.python import templater
from crypta.lib.python.yt import schema_utils
from crypta.profile.lib.frozen_dict import FrozenDict
import crypta.profile.runners.segments.lib.constructor_segments.common.utils as standard_utils
from crypta.profile.utils import (
    loggers,
    luigi_utils,
    utils,
    yql_utils,
)
from crypta.profile.utils.config import config


NORMALIZED_INPUT_TABLE_SCHEMA = schema_utils.yt_schema_from_dict({
    "id": "any",
    "id_type": "string",
    "rule_lab_id": "string",
})


@yt.with_context
class NormalizePrecalculatedTablesMapper(object):
    def __init__(self, table_index_to_table_rules,):
        self.table_index_to_table_rules = table_index_to_table_rules

    def __call__(self, row, context):
        table_rules = self.table_index_to_table_rules[context.table_index]
        for rule_id, rule_details in table_rules.items():
            id_column = rule_details["id_column"]
            if id_column in row:
                yield {
                    "id": row[id_column],
                    "id_type": str(rule_details["id_type"]),
                    "rule_lab_id": str(rule_id),
                }


def normalize_input_tables(yt_client, table_rules, output_table_path):
    yt_client.create_empty_table(
        output_table_path,
        schema=NORMALIZED_INPUT_TABLE_SCHEMA,
    )

    if not table_rules:
        return

    table_index_to_table_rules = {}
    input_table_paths = []
    for index, table_path_and_table_rules in enumerate(sorted(table_rules.items(), key=lambda x: x[0])):
        path, rules = table_path_and_table_rules
        input_table_paths.append(path)
        table_index_to_table_rules[index] = rules

    yt_client.run_map(
        NormalizePrecalculatedTablesMapper(table_index_to_table_rules),
        input_table_paths,
        output_table_path,
    )


class GetStandardSegmentsByPrecalculatedTables(luigi_utils.BaseYtTask):
    date = luigi.Parameter()
    task_group = 'constructor_segments'
    table_rules = luigi.Parameter(significant=False)
    rule_revision_ids = luigi.Parameter(significant=False)

    """
    Example of table_to_rule_lab_id:

    table_to_rule_lab_id = {
            '//home/crypta/team/terekhinam/test_upload': {
                'rule-bdb67b84': {
                    'id_column': 'yandexuid',
                    'id_type': 'yandexuid',
                    'update_interval': 1,
                },
                'rule-91612540': {
                    'id_column': 'yandexuid',
                    'id_type': 'yandexuid',
                    'update_interval': 1,
                },
            },
        }
    """

    priority = 100

    def requires(self):
        return {
            "direct_users": luigi_utils.ExternalInput(config.DIRECT_USERS),
            "gaid_cryptaid": luigi_utils.ExternalInput(utils.get_matching_table('gaid', 'crypta_id')),
            "idfa_cryptaid": luigi_utils.ExternalInput(utils.get_matching_table('idfa', 'crypta_id')),
        }

    def run(self):
        self.logger.info('Table to rule ids: {}'.format(dict(self.table_rules)))
        with loggers.TimeTracker(self.__class__.__name__), self.yt.Transaction() as transaction:
            self.compute(
                self.output().table,
                transaction,
            )

            self.yt.set_attribute(self.output().table, 'generate_date', self.date)
            self.yt.set_attribute(self.output().table, 'rule_ids', sorted(self.rule_revision_ids))

    def compute(self, output_table, transaction):
        self.yt.create_empty_table(
            output_table,
            schema=standard_utils.aggregated_schema,
        )

        if not self.table_rules:
            return

        with self.yt.TempTable() as normalized_input_table:
            normalize_input_tables(self.yt, self.table_rules, normalized_input_table)

            query_string = templater.render_resource("/query/precalculated_tables.yql", strict=True, vars={
                'input_table': normalized_input_table,
                'output_table': output_table,
                'direct_users': self.input()['direct_users'].table,
                'gaid_cryptaid': self.input()['gaid_cryptaid'].table,
                'idfa_cryptaid': self.input()['idfa_cryptaid'].table,
            })

            self.yql.query(
                query_string,
                udf_url_dict={'libcrypta_identifier_udf.so': yql_utils.get_udf_path(config.CRYPTA_IDENTIFIERS_UDF_PATH)},
                transaction=transaction,
            )

    def output(self):
        return luigi_utils.YtTableMultipleAttributeTarget(
            os.path.join(config.AGGREGATED_STANDARD_HEURISTIC_DIRECTORY, self.__class__.__name__),
            {
                'generate_date': self.date,
                'rule_ids': sorted(self.rule_revision_ids),
            }
        )

    @staticmethod
    def prepare_rules(rule_conditions, segments_config):
        table_rules = collections.defaultdict(lambda: collections.defaultdict(dict))
        rejected_rule_conditions = []

        for rule_condition in rule_conditions:
            try:
                rule_lab_id = segments_config.rule_revision_id_to_rule_id[rule_condition.revision]
                data = json.loads(rule_condition.values[0])

                if data['idType'] == 'yuid':
                    data['idType'] = 'yandexuid'

                if segments_config.yt.exists(data['path']):
                    segments_config.yt.get_attribute(data['path'], 'schema')

                    table_rules[data['path']][rule_lab_id] = {
                        'id_column': data['idKey'],
                        'id_type': data['idType'],
                        'update_interval': data['updateInterval'],
                    }
                else:
                    rejected_rule_conditions.append(rule_condition)

            except Exception as exception:
                if isinstance(exception, YtResponseError) and exception.is_access_denied():
                    # if got access_denied, get and put rule condition in order to write error to database
                    rejected_rule_conditions.append(rule_condition)
                    segments_config.logger.exception(u'Access denied for robot-crypta to %s (%s: %s)', data['path'], rule_lab_id, rule_condition.source)
                else:
                    segments_config.logger.exception(u'Error rule_id: %s, rule_condition.source: %s', rule_lab_id, rule_condition.source)

        return {"table_rules": FrozenDict(table_rules)}, rejected_rule_conditions
