import collections

import luigi

from crypta.lib.python import templater
from crypta.profile.lib.frozen_dict import FrozenDict
from crypta.profile.lib.segments import condition_helpers
from crypta.profile.runners.segments.lib.constructor_segments.common import (
    build_index,
    utils,
)
from crypta.profile.runners.segments.lib.constructor_segments.daily_rule_processors.words import word_rules
from crypta.profile.utils.config import config


class BuildSearchResultsRulesIndex(build_index.BuildIndex):
    query_resource = "/query/build_search_results_index.yql"


class GetStandardSegmentsBySearchResultsDayProcessor(utils.DailyRulesProcessor):
    host_to_rule_revision_id = luigi.Parameter(significant=False)
    input_dir = config.REQANS_PARSED_DIR
    index_dir = config.REQANS_HOSTS_INDEX_DIR

    def __init__(self, *args, **kwargs):
        super(GetStandardSegmentsBySearchResultsDayProcessor, self).__init__(*args, **kwargs)

    def requires(self):
        return BuildSearchResultsRulesIndex(
            date=self.date,
            input_dir=self.input_dir,
            index_dir=self.index_dir,
        )

    def compute(self, input_table, output_table, tx):
        if self.rule_revision_ids_to_be_prepared:
            host_to_rule_revision_id_filtered = {}
            for host, rule_revision_id_set in self.host_to_rule_revision_id.iteritems():
                filtered_rule_revision_ids = rule_revision_id_set.intersection(self.rule_revision_ids_to_be_prepared)
                if filtered_rule_revision_ids:
                    host_to_rule_revision_id_filtered[host] = filtered_rule_revision_ids
            self.logger.info('Host to rule ids to prepare: {}'.format(host_to_rule_revision_id_filtered))
            self.host_to_rule_revision_id = FrozenDict(host_to_rule_revision_id_filtered)

        self.yt.create_empty_table(
            output_table,
            schema=utils.daily_schema,
        )

        if self.host_to_rule_revision_id:
            self.yql.query(
                query_string=templater.render_resource("/query/search_results.yql", strict=True, vars={
                    "output_table": output_table,
                    "input_table": input_table,
                    "host_to_rule_revision_id": self.host_to_rule_revision_id,
                }),
                transaction=tx,
            )

    @classmethod
    def prepare_rules(cls, rule_conditions, segments_config):
        values_to_rule_revision_id = collections.defaultdict(set)

        for rule_condition in rule_conditions:
            for value in condition_helpers.normalize_hosts(rule_condition.values):
                values_to_rule_revision_id[value].add(rule_condition.revision)

        return {"host_to_rule_revision_id": FrozenDict(values_to_rule_revision_id)}, []


class GetStandardSegmentsBySearchRequestsDayProcessor(word_rules.DailyWordRulesProcessor):
    input_dir = config.REQANS_PARSED_DIR
    index_dir = config.REQANS_WORDS_INDEX_DIR
    yql_word_filter_data_size_per_job = "50M"
