import luigi
import yt.clickhouse as chyt

from crypta.profile.lib.segments import condition_helpers
from crypta.profile.runners.segments.lib.constructor_segments.common import (
    build_index,
    utils,
)
from crypta.profile.utils.config import config
from crypta.profile.utils.segment_utils.url_filter import UrlFilter


class BuildUrlRulesIndex(build_index.BuildIndex):
    query_resource = "/query/build_url_index.yql"


class DailyUrlRulesProcessor(utils.DailyRulesProcessor):
    url_filter = luigi.Parameter(significant=False)

    def requires(self):
        return BuildUrlRulesIndex(
            date=self.date,
            input_dir=self.input_dir,
            index_dir=self.index_dir,
        )

    def compute_pre_tx(self, input_table, output_table):
        self.tmp_table = self.yt.create_temp_table()
        self.yt.create_empty_table(self.tmp_table, schema={"yandexuid": "uint64", "rule_id": "uint64"})

        queries = self.url_filter.get_yql_queries(
            input_table=input_table,
            output_table=self.tmp_table,
            source=self.source,
            rule_revision_ids=self.rule_revision_ids_to_be_prepared,
        )
        for query in queries:
            self.logger.info("Query: %s", query)
            self.logger.info("Result: %s", list(chyt.execute(query, alias=config.CHYT_ALIAS, client=self.yt)))

    def compute(self, input_table, output_table, tx):
        self.yt.move(self.tmp_table, output_table, recursive=True, force=True)

    @classmethod
    def prepare_rules(cls, rule_conditions, segments_config):
        url_filter = UrlFilter(logger=segments_config.logger)

        for rule_condition in rule_conditions:
            rule_lab_id = segments_config.rule_revision_id_to_rule_id[rule_condition.revision]

            for url_condition in condition_helpers.split_urls(rule_condition.values):
                if url_condition.host not in segments_config.existing_hosts:
                    segments_config.logger.error('host {} does not exist ({})'.format(url_condition.host, rule_lab_id))
                    continue

                metrica_count, browser_count = segments_config.existing_hosts[url_condition.host]
                if (metrica_count if cls.is_metrica else browser_count) == 0:
                    continue

                url_filter.add_url(rule_condition.revision, url_condition.host, url_condition.path)

        return {"url_filter": url_filter}, []
