import datetime
import os
import uuid

from cached_property import cached_property
import luigi
import six

from crypta.lib.python.yt import yt_helpers
from crypta.profile.runners.segments.lib.constructor_segments.common import (
    build_index,
    utils,
)
from crypta.profile.utils.config import config
from crypta.profile.utils.segment_utils.yql_word_filter import YqlWordFilter


class BuildWordRulesIndex(build_index.BuildIndex):
    query_resource = "/query/build_word_index.yql"


class DailyWordRulesProcessor(utils.DailyRulesProcessor):
    yql_filter = luigi.Parameter(significant=False)

    def __init__(self, *args, **kwargs):
        super(DailyWordRulesProcessor, self).__init__(*args, **kwargs)
        self._query = None

    @cached_property
    def word_rules_file(self):
        name = str(uuid.uuid4())
        return os.path.join(config.PROFILES_TMP_WORD_RULES_YT_DIRECTORY, name)

    def requires(self):
        return BuildWordRulesIndex(
            date=self.date,
            input_dir=self.input_dir,
            index_dir=self.index_dir,
        )

    def compute_pre_tx(self, input_table, output_table):
        self._query = self.yql_filter.get_yql_query(
            input_table=input_table,
            output_table=output_table,
            word_rules_file='{proxy}/{path}'.format(proxy=config.CRYPTA_YT_PROXY, path=self.word_rules_file.lstrip('/')),
            rule_revision_ids=self.rule_revision_ids_to_be_prepared,
            data_size_per_job=self.yql_word_filter_data_size_per_job,
        )
        if self._query:
            with self.yt.Transaction():
                self.yt.create('file', path=self.word_rules_file, recursive=True, force=True)
                self.yt.write_file(self.word_rules_file, six.ensure_binary(self._query.word_conditions))
                yt_helpers.set_ttl(
                    yt_client=self.yt,
                    table=self.word_rules_file,
                    ttl_timedelta=datetime.timedelta(days=config.PROFILES_TMP_WORD_RULES_YT_TTL_DAYS),
                )

    def compute(self, input_table, output_table, tx):
        self.yt.create_empty_table(
            output_table,
            schema=utils.daily_schema,
        )

        if self.rule_revision_ids_to_be_prepared and self._query:
            self.yql.query(self._query.query_text, transaction=tx)

    @classmethod
    def prepare_rules(cls, rule_conditions, segments_config):
        yql_word_filter = YqlWordFilter(logger=segments_config.logger)

        for rule_condition in rule_conditions:
            checker = YqlWordFilter(logger=segments_config.logger)

            rule_lab_id = segments_config.rule_revision_id_to_rule_id[rule_condition.revision]
            checked_conditions = []

            for condition in rule_condition.values:
                try:
                    checker.parse_condition_string(condition)
                    checked_conditions.append(condition)
                except Exception:
                    segments_config.logger.exception(
                        'Incorrect condition: "%s" rule_revision_id=%s rule_lab_id=%s\n',
                        condition, rule_condition.revision, rule_lab_id,
                    )

            conditions = checked_conditions
            if not conditions:
                segments_config.logger.error('no valid conditions %s: %s', rule_condition.source, rule_condition)
                continue

            if len(conditions) > 1:
                condition_string = u' OR '.join([u'({})'.format(condition) for condition in conditions])
            else:
                condition_string = conditions[0]
            yql_word_filter.add_condition_string(rule_condition.revision, condition_string)

        return {"yql_filter": yql_word_filter}, []
