#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

import luigi

from crypta.lib.python import templater
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import BaseYtTask, YtNodeAttributeTarget
from crypta.profile.runners.segments.lib.constructor_segments.build_constructor_segments import BuildConstructorSegments


class GetLalSegments(BaseYtTask):
    date = luigi.Parameter()
    task_group = 'constructor_segments'
    lal_fixed_size = [500000, 1000000, 2000000, 4000000, 10000000]

    def requires(self):
        return BuildConstructorSegments(date=self.date)

    def output(self):
        return YtNodeAttributeTarget(
            config.REGULAR_LAL_INPUT_DIRECTORY,
            attribute_name=self.__class__.__name__,
            attribute_value=str(self.date),
        )

    def run(self):
        with self.yt.Transaction() as transaction:
            lal_config = self.yt.get_attribute(self.input().table, 'lal_config')
            self.logger.info('segment_config = {}'.format(lal_config))

            self.yql.query(
                query_string=templater.render_resource(
                    "/query/build_lal_core.yql",
                    strict=True,
                    vars={
                        'music_likes': os.path.join(config.AGGREGATED_STANDARD_HEURISTIC_DIRECTORY, 'GetStandardSegmentsByMusicLikes'),
                        'daily_logs_aggregator': os.path.join(config.AGGREGATED_STANDARD_HEURISTIC_DIRECTORY, 'DailyLogsAggregator'),
                        'precalculated_tables': os.path.join(config.AGGREGATED_STANDARD_HEURISTIC_DIRECTORY, 'GetStandardSegmentsByPrecalculatedTables'),
                        'mobile_apps': os.path.join(config.AGGREGATED_STANDARD_HEURISTIC_DIRECTORY, 'GetStandardSegmentsByMobileApp'),
                        'indevice_yandexuid': config.INDEVICE_YANDEXUID,
                        'cryptaid_yandexuid': config.CRYPTAID_YANDEXUID_TABLE,
                        'lookalike_input_dir': config.REGULAR_LAL_INPUT_DIRECTORY,
                        'segments_config': lal_config,
                    },
                ),
                transaction=transaction,
            )

            for lal_info in lal_config:
                table = os.path.join(config.REGULAR_LAL_INPUT_DIRECTORY, lal_info['segment_type'], str(lal_info['segment_id']))

                if self.yt.exists(table):
                    row_count = self.yt.get_attribute(table, 'row_count')

                    if row_count == 0:
                        self.logger.info("Source for lal segment_type: {}, segment_id: {} has size 0".format(lal_info['segment_type'], lal_info['segment_id']))
                        self.yt.remove(table)
                        continue

                    if 'max_coverage' in lal_info:
                        lal_info['max_coverage'] = min(lal_info['max_coverage'], config.MAX_LAL_SIZE)  # 50M is the max size of the audience segment

                    elif 'exponent' in lal_info:
                        if row_count < config.MIN_LAL_SIZE:
                            lal_info['exponent'] = min(len(self.lal_fixed_size) - 1, lal_info['exponent'])
                            lal_info['max_coverage'] = self.lal_fixed_size[lal_info['exponent']]
                        else:
                            lal_info['max_coverage'] = min(pow(2, lal_info['exponent']) * row_count, config.MAX_LAL_SIZE)  # 50M is the max size of the audience segment
                    else:
                        self.logger.info("No lal size info for: {}, segment_id: {}".format(lal_info['segment_type'], lal_info['segment_id']))
                        self.yt.remove(table)
                        continue

                    attrs_to_set = {
                        '_max_coverage': lal_info['max_coverage'],
                        '_include_input': lal_info['include_input'],
                        'crypta_related_goals': [],
                        'crypta_status': 'new',
                        'segment_priority': 1,
                        'segment_id': lal_info['segment_id'],
                        'crypta_maintain_device_distribution': False,
                        'crypta_maintain_geo_distribution': False,
                    }

                    for key, value in attrs_to_set.iteritems():
                        self.yt.set_attribute(table, key, value)

            self.yt.set_attribute(config.REGULAR_LAL_INPUT_DIRECTORY, self.__class__.__name__, str(self.date))
