#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Crypta user profile Validation processor

Usage:
  main.py validate_by_sources [--log-file-path=<path>]
  main.py custom_validation [--log-file-path=<path>]
  main.py (-h | --help)

Options:
  validate_by_sources           Run regular profiles validation by sources
  custom_validation             Run custom sample validation by sources
  --log-file-path=<path>        Path for logs
"""
import luigi
import datetime
from docopt import docopt

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config

from crypta.profile.tasks.monitoring.validation_by_sources.confusion_matrix import ComputeConfusionMatrix
from crypta.profile.tasks.monitoring.validation_by_sources.custom_validation import main as custom_validation
from crypta.profile.tasks.monitoring.validation_by_sources.make_charts import SendValidationResults


class ValidationTasks(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        today = self.date
        yesterday = date_helpers.get_yesterday(today)

        return [
            SendValidationResults(yesterday),
            ComputeConfusionMatrix(yesterday),
        ]


def main():
    arguments = docopt(__doc__)

    if arguments['--log-file-path']:
        config.LOGS_DIRECTORY = config.TASKS_LOGS_DIRECTORY = arguments['--log-file-path']
        config.LOCAL_STORAGE_DIRECTORY = arguments['--log-file-path']

    if arguments['validate_by_sources']:
        luigi.run(
            [
                '--scheduler-url', config.LUIGI_SCHEDULER_URL,
                '--workers', '1',
                '--date', str(datetime.date.today()),
                '--log-level', 'INFO',
            ],
            main_task_cls=ValidationTasks,
        )
    elif arguments['custom_validation']:
        custom_validation()


if __name__ == '__main__':
    main()
