from collections import defaultdict
import datetime

from crypta.lib.python.smtp.text_mail_sender import TextMailSender
from crypta.lib.python import (
    templater,
    time_utils,
)
from crypta.lib.python import swagger
from crypta.lib.python.yql import yql_helpers
from crypta.lib.python.yt import yt_helpers
from crypta.profile.services.disable_unused_segments.lib import templates

DISABLE_REQUESTED = "DISABLE_REQUESTED"


class AddresseeType:
    author = "author"
    responsible = "responsible"
    stakeholder = "stakeholder"


ADDRESSEE_TYPE_TO_TEMPLATE = {
    AddresseeType.author: templates.AUTHOR_TEMPLATE,
    AddresseeType.responsible: templates.RESPONSIBLE_TEMPLATE,
    AddresseeType.stakeholder: templates.STAKEHOLDER_TEMPLATE,
}


class ExportActions:
    def __init__(self):
        self.to_delete = []
        self.to_activate = []
        self.to_disable = []


def get_unused_exports(config, logger, yt_client):
    start_date = (datetime.datetime.fromtimestamp(time_utils.get_current_time()) - datetime.timedelta(days=config.UnusedPeriodDays)).strftime("%Y-%m-%d")

    with yt_client.Transaction() as tx, yt_client.TempTable() as tmp_table:
        params = {
            "start_date": start_date,
            "output_table": tmp_table,
            "input_dir": config.SegmentsStatsDir,
            "segments_simple": config.SegmentsSimpleTable,
        }
        yql_helpers.run_query("/query/calc_unused.yql", config.Yt, params, logger, tx=tx)

        return {(row["KeywordID"], row["SegmentID"]) for row in yt_client.read_table(tmp_table)}


def get_export_actions(unused_exports, api):
    actions = ExportActions()
    current_ts = time_utils.get_current_time()
    parents = api.lab.getParentsPerSegment().result()

    states_to_skip = {"DISABLED", "DELETED", "CREATED"}

    for export in api.lab.getExportsWithRuleId().result():
        segment_id = api.lab.getSegmentByExportId(id=export.id).result().id
        if export.nextActivityCheckTimestamp > current_ts or "root-crypta" in parents[segment_id] or export.state in states_to_skip:
            continue

        if (export.keywordId, export.segmentId) in unused_exports:
            if export.state == DISABLE_REQUESTED:
                # the waiting time exceeded
                actions.to_delete.append(export.id)
            else:
                actions.to_disable.append(export.id)
        else:
            if export.state == DISABLE_REQUESTED:
                actions.to_activate.append(export.id)
            api.lab.updateExportNextActivityCheckTs(id=export.id, timestamp=current_ts + datetime.timedelta(days=1).total_seconds()).result()

    return actions


def send_emails(authors, responsibles, stakeholders, mail_sender, logger):
    logger.info(
        "Sending emails to authors=%s, responsibles=%s, stakeholders=%s",
        authors,
        responsibles,
        stakeholders,
    )

    export_link = "https://lab.crypta.yandex-team.ru/segments?segment={}&export={}"

    for addressee_type, users_to_exports in [(AddresseeType.author, authors), (AddresseeType.responsible, responsibles), (AddresseeType.stakeholder, stakeholders)]:
        for user in users_to_exports:
            links = [export_link.format(segment_id, export_id) for segment_id, export_id in users_to_exports[user]]
            try:
                mail_sender.send(
                    subject=templates.SUBJECT,
                    text=templater.render_template(
                        ADDRESSEE_TYPE_TO_TEMPLATE.get(addressee_type),
                        vars={
                            "links": links,
                        }
                    ),
                    to_addrs=[user + "@yandex-team.ru"],
                )
            except Exception:
                logger.exception(
                    "Tried to send letter to %s about the following exports: %s, but caught exception with code=%s, message=%s.",
                    user,
                    links,
                )


def delete_exports(exports, logger, api):
    for export in exports:
        try:
            api.lab.deleteSegmentExport(id=export).result()
        except Exception:
            logger.exception(
                "Tried to delete export=%s but caught exception with code=%s, message=%s",
                export,
            )


def activate_exports(exports, logger, api):
    for export in exports:
        try:
            api.lab.updateExportState(id=export, state="ACTIVE").result()
        except Exception:
            logger.exception(
                "Tried to activate export=%s but caught exception with code=%s, message=%s",
                export,
            )


def request_disabling(exports, logger, api, config):
    for export in exports:
        try:
            api.lab.updateExportState(id=export, state=DISABLE_REQUESTED).result()
            api.lab.updateExportNextActivityCheckTs(id=export, timestamp=time_utils.get_current_time() + datetime.timedelta(days=config.WaitForReplyDays).total_seconds()).result()
        except Exception:
            logger.exception(
                "Tried to disable export=%s but caught exception with code=%s, message=%s",
                export,
            )


def collect_authors_responsibles_stakeholders(exports, api):
    authors_exports = defaultdict(list)
    responsibles_exports = defaultdict(list)
    stakeholders_exports = defaultdict(list)

    for export in exports:
        segment_attrs = api.lab.getSegmentByExportId(id=export).result()
        segment = api.lab.getSegment(id=segment_attrs.id).result()
        segment_export = (segment.id, export)

        if segment.author != "":
            authors_exports[segment.author].append(segment_export)

        for responsible in segment.responsibles:
            if responsible != segment.author:
                responsibles_exports[responsible].append(segment_export)

        for stakeholder in segment.stakeholders:
            if stakeholder != segment.author and stakeholder not in segment.responsibles:
                stakeholders_exports[stakeholder].append(segment_export)

    return authors_exports, responsibles_exports, stakeholders_exports


def disable_exports(exports, logger, api, config):
    mail_sender = TextMailSender.from_proto(config.Smtp)
    authors, responsibles, stakeholders = collect_authors_responsibles_stakeholders(exports, api)

    send_emails(authors, responsibles, stakeholders, mail_sender, logger)
    request_disabling(exports, logger, api, config)


def run(config, logger):
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)
    api = swagger.swagger(config.ApiConfig.Url, config.ApiConfig.Token)

    unused_exports = get_unused_exports(config, logger, yt_client)
    actions = get_export_actions(unused_exports, api)

    delete_exports(actions.to_delete, logger, api)
    activate_exports(actions.to_activate, logger, api)
    disable_exports(actions.to_disable, logger, api, config)
