import collections
import os

import six

from crypta.lib.python import swagger
from crypta.lib.python.yt import yt_helpers
from crypta.profile.runners.export_profiles.lib.profiles_generation import postprocess_profiles
from crypta.profile.services.exports_based_on_socdem_or_lal.lib import schema
from crypta.profile.utils.segment_utils import boolparser


ROOT_GROUP = u'root'
SOCDEM_GROUP = u'group-af00f55a'

RULE_CONDITION_SOURCE_TABLE = u'PRECALCULATED_TABLES'

SOCDEM_ID_TO_NAME = {
    u'segment-age_segment_0_17': u'0-17',
    u'segment-age_segment_18_24': u'18-24',
    u'segment-age_segment_25_34': u'25-34',
    u'segment-age_segment_35_44': u'35-44',
    u'segment-age_segment_45_54': u'45-54',
    u'segment-age_segment_55_99': u'55+',

    u'segment-96ac2e77': u'0-12',
    u'segment-777ccbab': u'13-17',
    u'segment-2c1b2e3e': u'18-20',

    u'segment-income_segment_a': u'income_a',
    u'segment-income_segment_b1': u'income_b1',
    u'segment-income_segment_b2': u'income_b2',
    u'segment-income_segment_c1': u'income_c1',
    u'segment-income_segment_c2': u'income_c2',

    u'segment-gender_female': u'female',
    u'segment-gender_male': u'male',
}

CATEGORY_SOCDEM = u'socdem'
CATEGORY_LAL = u'lal'
CATEGORY_CODE = u'code'
CATEGORY_YT_TABLE = u'yt_table'
CATEGORY_RULE = u'rule'
CATEGORY_DERIVATIVE = u'derivative'

RULE_STATUS_EMPTY = 0
RULE_STATUS_WITH_TABLE = 1
RULE_STATUS_TABLE_FREE = 2


FieldsNames = collections.namedtuple('FieldsNames', ['based_on_field', 'sources_field'])


CATEGORY_TO_FIELDS = {
    CATEGORY_SOCDEM: FieldsNames(schema.IS_BASED_ON_SOCDEM, schema.SOURCES_SOCDEM),
    CATEGORY_LAL: FieldsNames(schema.IS_BASED_ON_LAL, schema.SOURCES_LAL),
    CATEGORY_CODE: FieldsNames(schema.IS_BASED_ON_CODE, schema.SOURCES_CODE),
    CATEGORY_YT_TABLE: FieldsNames(schema.IS_BASED_ON_YT_TABLE, schema.SOURCES_YT_TABLE),
}


Category = collections.namedtuple('Category', ['prime_category', 'additional_category'])


def get_api_token():
    api_token = os.environ.get('API_TOKEN')
    assert api_token is not None, "Can't find environment variable API_TOKEN"

    return api_token


def make_parent_id_dict(segments, groups):
    parent_ids = {}
    for segment in segments:
        parent_ids[segment.id] = segment.parentId
        for export in segment.exports.exports:
            parent_ids[export.id] = segment.id

    for group in groups:
        parent_ids[group.id] = group.parentId

    return parent_ids


def is_in_socdem_group(obj, id_to_parent_id):
    obj_id = obj.id

    while obj_id not in (ROOT_GROUP, SOCDEM_GROUP):
        obj_id = id_to_parent_id[obj_id]

    return obj_id == SOCDEM_GROUP


def get_rule_status(rule_id, rules_dict):
    if not rule_id:
        return RULE_STATUS_EMPTY

    rule = rules_dict[rule_id]
    if not rule.conditions:
        return RULE_STATUS_EMPTY

    if any(map(lambda condition: condition.source == RULE_CONDITION_SOURCE_TABLE, rule.conditions)):
        return RULE_STATUS_WITH_TABLE
    return RULE_STATUS_TABLE_FREE


def get_category(export, export_trees, rules_dict, id_to_parent_id):
    export_expression = export_trees[export.id]

    if export.id not in export_expression.dependencies:
        return Category(CATEGORY_DERIVATIVE, None)

    has_expression = not isinstance(export_expression, boolparser.LeafNode)
    rule_status = get_rule_status(export.ruleId, rules_dict)

    if is_in_socdem_group(export, id_to_parent_id) and not has_expression:
        category = CATEGORY_SOCDEM
    elif export.lal:
        category = CATEGORY_LAL
    elif rule_status == RULE_STATUS_WITH_TABLE:
        category = CATEGORY_YT_TABLE
    elif rule_status == RULE_STATUS_TABLE_FREE:
        category = CATEGORY_RULE
    else:
        category = CATEGORY_CODE

    prime_category, additional_category = category, None
    if has_expression:
        prime_category, additional_category = CATEGORY_DERIVATIVE, prime_category
    return Category(prime_category, additional_category)


def take_dependency_into_account(current_row, dependency_row):
    for category, (based_on_field, sources_field) in CATEGORY_TO_FIELDS.items():
        if dependency_row[schema.CATEGORY] == category:
            current_row[based_on_field] = True
            current_row[sources_field].add(dependency_row[schema.EXPORT_ID])

        if dependency_row[based_on_field]:
            current_row[based_on_field] = True
            current_row[sources_field].update(dependency_row[sources_field], [dependency_row[schema.EXPORT_ID]])


def describe_export(current_id, export_rows, export_trees, exports_dict, rules_dict, id_to_parent_id):
    export = exports_dict[current_id]
    current_row = export_rows[current_id]

    category = get_category(export, export_trees, rules_dict, id_to_parent_id)
    current_row[schema.CATEGORY] = category.prime_category

    deps = export_trees[current_id].dependencies
    for dependency_id in deps:
        if dependency_id != current_id:
            dependency_row = export_rows[dependency_id]
            if dependency_row[schema.CATEGORY] is None:
                describe_export(dependency_id, export_rows, export_trees, exports_dict, rules_dict, id_to_parent_id)

            take_dependency_into_account(current_row, dependency_row)

    if CATEGORY_TO_FIELDS.get(category.additional_category):
        based_on_field, sources_field = CATEGORY_TO_FIELDS[category.additional_category]
        current_row[based_on_field] = True
        current_row[sources_field].add(current_id)

    current_row[schema.HUMAN_READABLE_SOCDEM_NAMES] = set(
        filter(
            None,
            [SOCDEM_ID_TO_NAME.get(id_to_parent_id[export_id]) for export_id in current_row[schema.SOURCES_SOCDEM]],
        ),
    )


def make_empty_row(export):
    row = {
        schema.EXPORT_ID: export.id,
        schema.KEYWORD_ID: export.keywordId,
        schema.SEGMENT_ID: export.segmentId,
        schema.CATEGORY: None,
        schema.HUMAN_READABLE_SOCDEM_NAMES: set(),
    }

    for based_on_field, sources_field in CATEGORY_TO_FIELDS.values():
        row[based_on_field] = False
        row[sources_field] = set()

    return row


def write_to_ml_table(export_rows, config):
    yt_client = yt_helpers.get_yt_client(config.yt_proxy)

    to_table = list(six.itervalues(export_rows))

    for row in to_table:
        for sources_field in map(lambda name: name.sources_field, CATEGORY_TO_FIELDS.values()):
            row[sources_field] = list(row[sources_field])
        row[schema.HUMAN_READABLE_SOCDEM_NAMES] = list(row[schema.HUMAN_READABLE_SOCDEM_NAMES])

    to_table.sort(key=lambda row: row[schema.EXPORT_ID])

    yt_client.create(
        'table',
        path=config.output_table_path,
        recursive=True,
        force=True,
        attributes={'schema': schema.get_ml_table_schema()},
    )

    yt_client.write_table(config.output_table_path, to_table)


def run(config, logger):
    api = swagger.swagger(config.api_url, get_api_token())
    segments = api.lab.getAllSegments().result()
    groups = api.lab.getAllSegmentGroups().result()
    rules = api.lab.getAllRules().result()
    logger.info(
        'Got segments info from api. Segments number: {}, groups number: {}, rules number: {}'.format(
            len(segments),
            len(groups),
            len(rules),
        )
    )

    parser = boolparser.ExpressionParser(postprocess_profiles.get_export_expressions(segments), logger)
    export_trees = parser.build_trees()

    exports_dict = {
        export.id: export
        for segment in segments
        for export in segment.exports.exports
    }
    export_rows = {export_id: make_empty_row(exports_dict[export_id]) for export_id in export_trees}

    id_to_parent_id = make_parent_id_dict(segments, groups)
    rules_dict = {rule.id: rule for rule in rules}

    for export_id, row in six.iteritems(export_rows):
        if row[schema.CATEGORY] is None:
            describe_export(export_id, export_rows, export_trees, exports_dict, rules_dict, id_to_parent_id)

    write_to_ml_table(export_rows, config)
    logger.info('Output is written to {}'.format(config.output_table_path))
