import collections
import itertools
import os

import six
import six.moves
import sympy
from sympy.logic import boolalg

from crypta.lib.python import swagger
from crypta.lib.python.audience import id_converter
from crypta.lib.python.yt import yt_helpers
from crypta.profile.runners.export_profiles.lib.profiles_generation import postprocess_profiles
from crypta.profile.services.socdem_expressions_for_direct.lib import schema
from crypta.profile.utils.segment_utils import boolparser


SOCDEM_ID_TO_NAME = {
    u'segment-age_segment_0_17': u'age_18_minus',
    u'segment-age_segment_18_24': u'age_18_24',
    u'segment-age_segment_25_34': u'age_25_34',
    u'segment-age_segment_35_44': u'age_35_44',
    u'segment-age_segment_45_54': u'age_45_54',
    u'segment-age_segment_55_99': u'age_55_plus',

    u'segment-income_segment_a': u'income_a',
    u'segment-income_segment_b1': u'income_b1',
    u'segment-income_segment_b2': u'income_b2',
    u'segment-income_segment_c1': u'income_c1',
    u'segment-income_segment_c2': u'income_c2',

    u'segment-gender_female': u'gender_female',
    u'segment-gender_male': u'gender_male',
}

REQUIRED_KEYWORD = 557


SocdemPossibilities = collections.namedtuple(
    'SocdemPossibilities',
    ['socdem_symbols', 'possibilities', 'fully_evaluated'],
)


class Sympifier(object):
    def __init__(self, export_trees, export_id_to_segment_id):
        self.export_trees = export_trees
        self.export_id_to_segment_id = export_id_to_segment_id
        self._symbols = {}
        self._ready_expressions = {}

    def get_sympy_expression(self, export_id):
        ready = self._ready_expressions.get(export_id)
        if ready is not None:
            return ready

        expression = self._build_sympy_expression(self.export_trees[export_id], current_export=export_id)

        self._ready_expressions[export_id] = expression
        return expression

    def _build_sympy_expression(self, node, current_export):
        if isinstance(node, boolparser.LeafNode):
            if node.export_id_to_evaluate == current_export:
                name = self._get_socdem_name(current_export) or current_export
                return self._get_sympy_symbol(name)

            return self.get_sympy_expression(node.export_id_to_evaluate)

        if isinstance(node, boolparser.NotNode):
            return boolalg.Not(self._build_sympy_expression(node.child, current_export))

        if isinstance(node, boolparser.AndNode):
            return boolalg.And(*(self._build_sympy_expression(child, current_export) for child in node.children))

        if isinstance(node, boolparser.OrNode):
            return boolalg.Or(*(self._build_sympy_expression(child, current_export) for child in node.children))

    def _get_socdem_name(self, export_id):
        return SOCDEM_ID_TO_NAME.get(self.export_id_to_segment_id[export_id])

    def _get_sympy_symbol(self, name):
        return self._symbols.setdefault(name, sympy.symbols(name))


def get_api_token():
    api_token = os.environ.get('API_TOKEN')
    assert api_token is not None, "Can't find environment variable API_TOKEN"

    return api_token


def extract_exports(segments):
    return (export for segment in segments for export in segment.exports.exports)


def make_export_to_segment_dict(segments):
    return {
        export.id: segment.id
        for segment in segments
        for export in segment.exports.exports
    }


def get_socdem_possibilities(expression, max_vars_to_evaluate):
    socdem_symbols = tuple(
        symbol
        for symbol in expression.free_symbols
        if not symbol.name.startswith(u'export-')
    )

    if not socdem_symbols:
        return SocdemPossibilities(socdem_symbols=tuple(), possibilities={}, fully_evaluated=True)

    genders = [None]
    ages = [None]
    incomes = [None]
    for symbol in socdem_symbols:
        if symbol.name.startswith(u'gender_'):
            genders.append(symbol)
        elif symbol.name.startswith(u'age_'):
            ages.append(symbol)
        elif symbol.name.startswith(u'income_'):
            incomes.append(symbol)

    possibilities = {}
    fully_evaluated = True

    for current_socdem in itertools.product(genders, ages, incomes):
        socdem_values = tuple(symbol in current_socdem for symbol in socdem_symbols)
        expression_without_socdem = expression.subs(list(six.moves.zip(socdem_symbols, socdem_values)))

        if isinstance(expression_without_socdem, (boolalg.BooleanFalse, boolalg.BooleanTrue)):
            possibilities[socdem_values] = bool(expression_without_socdem)
            continue

        rest_symbols = list(expression_without_socdem.free_symbols)
        if len(rest_symbols) > max_vars_to_evaluate:
            possibilities[socdem_values] = True
            fully_evaluated = False
            continue

        possibilities[socdem_values] = False
        for values in itertools.product((False, True), repeat=len(rest_symbols)):
            if expression_without_socdem.subs(list(six.moves.zip(rest_symbols, values))):
                possibilities[socdem_values] = True
                break

    return SocdemPossibilities(socdem_symbols, possibilities, fully_evaluated)


def make_output_expression(socdem_possibilities):
    disjunctions = []
    found_zeros = set()

    for or_length in range(1, len(socdem_possibilities.socdem_symbols) + 1):
        for indices in itertools.combinations(range(len(socdem_possibilities.socdem_symbols)), or_length):
            is_candidate = True
            new_zeros = []
            for socdem_values, result in six.iteritems(socdem_possibilities.possibilities):
                current_socdem_values = tuple(socdem_values[i] for i in indices)
                if not any(current_socdem_values):
                    if result:
                        is_candidate = False
                        break
                    if socdem_values not in found_zeros:
                        new_zeros.append(socdem_values)

            if is_candidate and new_zeros:
                disjunctions.append(boolalg.Or(*(socdem_possibilities.socdem_symbols[i] for i in indices)))
                for to_zero in new_zeros:
                    found_zeros.add(to_zero)

    for not_idx in range(len(socdem_possibilities.socdem_symbols)):
        is_candidate = True
        new_zeros = []
        for socdem_values, result in six.iteritems(socdem_possibilities.possibilities):
            if socdem_values[not_idx]:
                if result:
                    is_candidate = False
                    break
                if socdem_values not in found_zeros:
                    new_zeros.append(socdem_values)

        if is_candidate and new_zeros:
            disjunctions.append(boolalg.Not(socdem_possibilities.socdem_symbols[not_idx]))
            for to_zero in new_zeros:
                found_zeros.add(to_zero)

    fits_perfectly = all(
        result or socdem_values in found_zeros
        for socdem_values, result in six.iteritems(socdem_possibilities.possibilities),
    )

    return boolalg.And(*disjunctions), fits_perfectly


def convert_to_string(expression):
    if isinstance(expression, boolalg.Not):
        return u'NOT {}'.format(expression.args[0].name)

    if isinstance(expression, boolalg.Or):
        return u' OR '.join(map(convert_to_string, expression.args))

    if isinstance(expression, boolalg.And):
        mults = []
        for arg in expression.args:
            if isinstance(arg, boolalg.Or):
                mults.append(u'({})'.format(convert_to_string(arg)))
            else:
                mults.append(convert_to_string(arg))
        return u' AND '.join(mults)

    return expression.name


def build_sympy_expressions(exports_dict, export_trees, export_id_to_segment_id, logger):
    sympifier = Sympifier(export_trees, export_id_to_segment_id)

    expressions = {
        export_id: sympifier.get_sympy_expression(export_id)
        for export_id in export_trees
        if exports_dict[export_id].keywordId == REQUIRED_KEYWORD
    }
    logger.info('Expressions are built for {} exports.'.format(len(expressions)))
    return expressions


def evaluate_socdem_possibilities(expressions, max_vars_to_evaluate, logger):
    exports_possibilities = {}
    not_fully_evaluated = 0
    include_all = 0
    include_nothing = 0
    without_socdem = 0
    for export_id, expression in six.iteritems(expressions):
        socdem_possibilities = get_socdem_possibilities(expression, max_vars_to_evaluate)

        if not socdem_possibilities.fully_evaluated:
            not_fully_evaluated += 1

        if not socdem_possibilities.socdem_symbols:
            without_socdem += 1
        elif all(six.itervalues(socdem_possibilities.possibilities)):
            include_all += 1
        elif not any(six.itervalues(socdem_possibilities.possibilities)):
            include_nothing += 1
        else:
            exports_possibilities[export_id] = socdem_possibilities

    logger.info("{} exports weren't fully evaluated due to big number of dependencies.".format(not_fully_evaluated))
    logger.info("{} exports don't depend on socdem.".format(without_socdem))
    logger.info("{} exports allow any socdem.".format(include_all))
    logger.info("{} exports don't allow any socdem.".format(include_nothing))
    logger.info("Truth tables are built for {} exports.".format(len(exports_possibilities)))

    return exports_possibilities


def build_output_strings(exports_possibilities, logger):
    strings = {}
    not_perfectly_fit = 0
    turned_to_true = 0
    for export_id, socdem_possibilities in six.iteritems(exports_possibilities):
        expr, fits = make_output_expression(socdem_possibilities)

        if not fits:
            not_perfectly_fit += 1

        if isinstance(expr, boolalg.BooleanTrue):
            turned_to_true += 1
        else:
            strings[export_id] = convert_to_string(expr)

    if not_perfectly_fit:
        logger.info(
            "It's impossible to build expression for {} truth tables. {} of them turned to tautology.".format(
                not_perfectly_fit,
                turned_to_true,
            ),
        )
    logger.info("Output strings are built for {} exports.".format(len(strings)))

    return strings


def write_to_table(config, logger, strings, exports_dict):
    yt_client = yt_helpers.get_yt_client(config.yt_proxy)

    to_table = [
        {
            schema.GOAL_ID: id_converter.segment_id_to_goal_id(exports_dict[export_id].segmentId),
            schema.SOCDEM: string,
            schema.EXPORT_ID: export_id,
        }
        for export_id, string in six.iteritems(strings)
    ]

    yt_client.create(
        'table',
        path=config.output_table_path,
        recursive=True,
        force=True,
        attributes={'schema': schema.get_table_schema()},
    )
    yt_client.write_table(config.output_table_path, to_table)
    logger.info('Output table is written to {}'.format(config.output_table_path))


def build_socdem_expressions(config, logger):
    api = swagger.swagger(config.api_url, get_api_token())
    segments = api.lab.getAllSegments().result()
    logger.info('Got segments info from api. Segments number: {}'.format(len(segments)))

    exports_dict = {export.id: export for export in extract_exports(segments)}
    export_id_to_segment_id = make_export_to_segment_dict(segments)

    export_trees = (boolparser.ExpressionParser(postprocess_profiles.get_export_expressions(segments), logger)
                    .build_trees())
    logger.info('Export trees are built. Exports number: {}'.format(len(export_trees)))

    expressions = build_sympy_expressions(exports_dict, export_trees, export_id_to_segment_id, logger)
    exports_possibilities = evaluate_socdem_possibilities(expressions, config.max_vars_to_evaluate, logger)
    strings = build_output_strings(exports_possibilities, logger)

    write_to_table(config, logger, strings, exports_dict)
