import json
import os

from crypta.lib.python import templater
from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.profile.lib.socdem_helpers import socdem_config
import crypta.profile.lib.socdem_helpers.tools.features as features_utils
from crypta.profile.lib.socdem_helpers.train_utils.train_helper import get_flat_cat_features_dict
from crypta.profile.utils.config import config

categorical_features_matching_query_template = """
$data = (
    SELECT
        id,
        id_type,
        model,
        manufacturer,
        categories,
        Yson::ConvertToUint64(main_region_obl) AS main_region_obl
    FROM `{{app_metrica_table}}`
);

{% for field_parameters in fields_parameters %}

$most_common = (
    SELECT
        {{field_parameters.name}},
        COUNT(*) AS cnt
    FROM $data
    WHERE {{field_parameters.name}} IS NOT NULL
    GROUP BY {{field_parameters.name}}
    ORDER BY cnt DESC
    LIMIT {{field_parameters.top_count}}
);

INSERT INTO `{{field_parameters.output_table}}`
WITH TRUNCATE

SELECT
    {{field_parameters.name}} AS feature,
    ROW_NUMBER() OVER w AS feature_index
FROM (
        SELECT *
        FROM $most_common
    )
WINDOW w AS (ORDER BY cnt DESC)
ORDER BY feature_index;

{% endfor %}

$categories_table = (
    SELECT Yson::ConvertToStringList(raw_categories) as raw_categories
    FROM `{{categories_table}}`
);

$categories_table = (
    SELECT DISTINCT category
    FROM $categories_table
    FLATTEN BY raw_categories AS category
);

INSERT INTO `{{categories_features_table}}`
WITH TRUNCATE

SELECT
    category AS feature,
    ROW_NUMBER() OVER w AS feature_index
FROM $categories_table
WINDOW w AS (ORDER BY category DESC)
ORDER BY feature_index;
"""


region_matching_template = """
INSERT INTO `{output_table_regions}` WITH TRUNCATE
SELECT
    feature,
    Geo::RegionById(CAST(feature as INT32)).en_name as en_name
FROM `{input_table_regions}`
"""


def get_table_path_for_feature(feature, folder=config.PRESTABLE_CATEGORICAL_FEATURES_MATCHING_DIR):
    return os.path.join(folder, feature)


def get(yt_client, yql_client, date, output_dict):
    with NirvanaTransaction(yt_client) as transaction:
        fields_parameters = []
        for field in ('model', 'manufacturer', 'main_region_obl'):
            fields_parameters.append({
                'name': field,
                'top_count': socdem_config.MOBILE_FEATURES_NUMBERS[field],
                'output_table': get_table_path_for_feature(field),
            })

        categorical_features_matching_query = templater.render_template(
            template_text=categorical_features_matching_query_template,
            vars={
                'app_metrica_table': config.APP_BY_DEVID_MONTHLY_TABLE,
                'fields_parameters': fields_parameters,
                'categories_table': config.MOBILE_APP_CATEGORIES,
                'categories_features_table': get_table_path_for_feature('categories'),
            },
        )

        yql_client.execute(
            query=categorical_features_matching_query,
            transaction=str(transaction.transaction_id),
            title='YQL calculate mobile categorical features matching',
        )

        yql_client.execute(
            query=region_matching_template.format(
                input_table_regions=get_table_path_for_feature('main_region_obl'),
                output_table_regions=get_table_path_for_feature('region_matching'),
            ),
            transaction=str(transaction.transaction_id),
            title='YQL calculate mobile categorical features matching (regions)',
        )

        for feature in features_utils.cat_feature_mobile_types:
            yt_client.set_attribute(get_table_path_for_feature(feature), 'generate_date', date)
        yt_client.set_attribute(get_table_path_for_feature('region_matching'), 'generate_date', date)

        cat_features_dicts, _ = features_utils.download_cat_features_to_dict(
            yt=yt_client,
            yt_folder_path=config.PRESTABLE_CATEGORICAL_FEATURES_MATCHING_DIR,
            is_mobile=True,
        )
        flat_features_dict = get_flat_cat_features_dict(cat_features_dicts, features_utils.cat_feature_mobile_types)

        with open(output_dict, 'w') as dict_file:
            json.dump(flat_features_dict, dict_file)
