from datetime import timedelta as dt

from crypta.lib.python import crypta_env
from crypta.lib.python.data_size import DataSize
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import (
    aggregate_hosts,
    consts,
    juggler_check_generator,
)
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from crypta.lib.python.spine.yt import yt_config_registry
from crypta.lib.python.spine.yt.yt_output_table_latency_metric import YtOutputTableLatencyMetric
from crypta.lib.python.spine.yt.yt_size_metric import YtSizeMetric
from crypta.profile.spine import profile_spine_helpers
from sandbox.projects.crypta import merge_to_bigb_collector


COMMON_AGGR_HOSTS = aggregate_hosts.AggregateHosts(testing_host="crypta.profile.testing", production_host="crypta.profile.production")


def get_registry():
    juggler = juggler_check_generator.CryptaYtCheckGenerator(tags=["crypta-profile"])

    _add_schedulers(juggler)
    _add_testing_juggler_checks(juggler)
    _add_production_juggler_checks(juggler)
    _add_output_table_latencies(juggler)
    _add_error_dirs_check(juggler)

    return juggler


def _add_schedulers(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler.clone(yt_dependencies=YtProxy.Group.offline), ["CRYPTA_PROFILE"])

    sandbox.create_run_universal_bundle_scheduler(
        bundle_name="profile-calc-lab-segments-stats",
        cmd=["{{cwd}}/crypta-profile-calc-lab-segments-stats", "--config", "{{cwd}}/config.yaml"],
        env=environment.STABLE,
        schedule_interval=dt(days=1),
        kill_timeout=dt(hours=12),
        retry_interval=dt(hours=1),
        sequential_run=True,
    ).check(crit_time=dt(days=2)).set_host(COMMON_AGGR_HOSTS.production_host.host)

    for aggr_host in COMMON_AGGR_HOSTS:
        sandbox.create_run_universal_bundle_scheduler(
            bundle_name="profile-upload-direct-exports-tanker-names-to-yt",
            cmd=["{{cwd}}/crypta-profile-upload-direct-exports-tanker-names-to-yt", "--config", "{{cwd}}/config.yaml"],
            env=aggr_host.environment,
            schedule_interval=dt(hours=2),
            kill_timeout=dt(hours=4),
            retry_interval=dt(hours=1),
            sequential_run=True,
            secrets_env=crypta_env.to_sandbox_secrets_env(crypta_env.crypta_api_oauth),
        ).check(crit_time=dt(days=2)).set_host(aggr_host.host)

        sandbox.create_run_universal_bundle_scheduler(
            bundle_name="profile-exports-based-on-socdem-or-lal",
            cmd=["{{cwd}}/crypta-profile-exports-based-on-socdem-or-lal", "--config", "{{cwd}}/config.yaml"],
            env=aggr_host.environment,
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=6),
            retry_interval=dt(hours=1),
            sequential_run=True,
            secrets_env={
                "API_TOKEN": "sec-01csvzhx0pk7094n7hkpvd73qt[token]",
            },
        ).check(crit_time=dt(days=2)).set_host(aggr_host.host)

        sandbox.create_run_universal_bundle_scheduler(
            bundle_name="profile-socdem-expressions-for-direct",
            cmd=["{{cwd}}/crypta-profile-socdem-expressions-for-direct", "--config", "{{cwd}}/config.yaml"],
            env=aggr_host.environment,
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=6),
            retry_interval=dt(hours=1),
            sequential_run=True,
            secrets_env={
                "API_TOKEN": "sec-01csvzhx0pk7094n7hkpvd73qt[token]",
            },
        ).check(crit_time=dt(days=2)).set_host(aggr_host.host)

        sandbox.create_run_universal_bundle_scheduler(
            bundle_name="profile-disable-unused-segments",
            cmd=["{{cwd}}/crypta-profile-disable-unused-segments", "--config", "{{cwd}}/config.yaml"],
            env=aggr_host.environment,
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=4),
            retry_interval=dt(hours=1),
            sequential_run=True,
            secrets_env=crypta_env.to_sandbox_secrets_env(crypta_env.crypta_api_token),
            additional_env={"CRYPTA_ENVIRONMENT": environment.SANDBOX_ENV_TO_CRYPTA_ENV[aggr_host.environment]}
        ).check(crit_time=dt(days=2)).set_host(aggr_host.host)

    for segment_group, kill_timeout, crit_time, schedule_interval, retry_interval in [
        ("electrocars", dt(hours=4), dt(days=2), dt(days=1), dt(hours=1)),
        ("travel", dt(hours=4), dt(days=3), dt(days=1), dt(hours=1)),
        ("zodiac_signs", dt(hours=1), dt(days=8), dt(days=7), dt(days=1)),
        ("travelers_to_moscow_with_night_stay", dt(hours=4), dt(days=8), dt(days=7), dt(days=1)),
        ("uslugi_contractors", dt(hours=4), dt(days=2), dt(days=1), dt(hours=1)),
    ]:
        create_precalculate_tables_scheduler(
            sandbox,
            segment_group,
            kill_timeout=kill_timeout,
            crit_time=crit_time,
            schedule_interval=schedule_interval,
            retry_interval=retry_interval,
        )

    for task in [
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-monitoring",
            binary="monitoring",
            schedule_daily_start_time="2018-12-05T06:30:00Z",
            retry_interval=dt(hours=2),
            kill_timeout=dt(hours=23),
            crit_time=dt(days=2),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-monitoring",
            binary="monitoring",
            subprogram="consistency",
            schedule_interval=dt(minutes=30),
            kill_timeout=dt(hours=3),
            crit_time=dt(days=2),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-log-parsing",
            binary="log_parsing",
            subprogram="make_sample",
            schedule_daily_start_time="2018-12-04T22:00:00Z",
            kill_timeout=dt(hours=3),
            retry_interval=dt(hours=1),
            crit_time=dt(days=1),
            aggr_hosts=[COMMON_AGGR_HOSTS.testing_host],
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-export-profiles",
            binary="export_profiles",
            subprogram="clean_up_tmp",
            schedule_interval=dt(hours=1),
            kill_timeout=dt(hours=1),
            retry_interval=dt(hours=1),
            crit_time=dt(days=1),
            aggr_hosts=COMMON_AGGR_HOSTS,
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-export-profiles",
            binary="export_profiles",
            schedule_daily_start_time="2020-07-29T21:05:00Z",
            kill_timeout=dt(hours=23),
            retry_interval=dt(minutes=20),
            crit_time=dt(days=2),
            aggr_hosts=[COMMON_AGGR_HOSTS.testing_host],
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-validation",
            binary="validation",
            subprogram="custom_validation",
            schedule_interval=dt(minutes=10),
            kill_timeout=dt(hours=6),
            retry_interval=dt(hours=1),
            crit_time=dt(hours=12),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-validation",
            binary="validation",
            subprogram="validate_by_sources",
            schedule_interval=dt(hours=4),
            kill_timeout=dt(hours=10),
            retry_interval=dt(hours=1),
            crit_time=dt(days=1),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-interests",
            binary="interests",
            subprogram="interests_tasks",
            kill_timeout=dt(hours=21),
            schedule_interval=dt(hours=1),
            crit_time=dt(days=2),
            aggr_hosts=COMMON_AGGR_HOSTS,
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-interests",
            binary="interests",
            subprogram="interests_log_processors_task",
            schedule_interval=dt(minutes=10),
            kill_timeout=dt(hours=23),
            crit_time=dt(days=2),
            aggr_hosts=COMMON_AGGR_HOSTS,
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-interests",
            binary="interests",
            subprogram="upload_shortterm_interests",
            schedule_interval=dt(minutes=5),
            kill_timeout=dt(hours=3),
            crit_time=dt(hours=12),
            aggr_hosts=COMMON_AGGR_HOSTS,
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-segments",
            binary="segments",
            subprogram="coded",
            kill_timeout=dt(hours=47),
            schedule_interval=dt(hours=1),
            crit_time=None,
            aggr_hosts=COMMON_AGGR_HOSTS,
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-segments",
            binary="segments",
            subprogram="constructor",
            schedule_interval=dt(minutes=60),
            kill_timeout=dt(hours=47),
            retry_interval=dt(minutes=20),
            crit_time=dt(days=2),
            aggr_hosts=COMMON_AGGR_HOSTS,
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-segments",
            binary="segments",
            subprogram="trainable",
            schedule_interval=dt(hours=1),
            kill_timeout=dt(hours=47),
            crit_time=None,
            aggr_hosts=COMMON_AGGR_HOSTS,
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-lal-samples",
            binary="lal_samples",
            schedule_interval=dt(hours=1),
            kill_timeout=dt(hours=24),
            retry_interval=dt(minutes=20),
            crit_time=dt(days=1),
            aggr_hosts=COMMON_AGGR_HOSTS,
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-export-profiles",
            binary="export_profiles",
            schedule_daily_start_time="2018-11-14T21:05:00Z",
            kill_timeout=dt(hours=36),
            retry_interval=dt(minutes=20),
            sequential_run=False,
            use_semaphore=False,
            crit_time=dt(hours=28),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
            requirements={"disk_space": 2 << 30},
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-export-profiles",
            binary="export_profiles",
            subprogram="precalculate_affinity",
            kill_timeout=dt(hours=15),
            schedule_interval=dt(hours=1),
            crit_time=dt(days=1),
            aggr_hosts=COMMON_AGGR_HOSTS,
        ),
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-export-profiles",
            binary="export_profiles",
            subprogram="upload_segments_to_audience",
            schedule_interval=dt(hours=1),
            kill_timeout=dt(hours=15),
            crit_time=dt(days=1),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
        ),
    ] + [
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-log-parsing",
            binary="log_parsing",
            subprogram=subprogram,
            schedule_interval=schedule_interval,
            kill_timeout=kill_timeout,
            crit_time=dt(hours=12),
            aggr_hosts=COMMON_AGGR_HOSTS,
        ) for subprogram, schedule_interval, kill_timeout in [
            ("geocube", dt(hours=1), dt(hours=3)),
            ("bb", dt(minutes=10), dt(hours=5)),
            ("metrics", dt(minutes=10), dt(hours=5)),
            ("bar", dt(minutes=10), dt(hours=3)),
            ("reqans", dt(minutes=10), dt(hours=3)),
            ("pp", dt(minutes=10), dt(hours=5)),
            ("mobile_bar", dt(minutes=10), dt(hours=3)),
        ]
    ]:
        profile_spine_helpers.create_run_universal_bundle_schedulers(sandbox, task)

    for env in environment.ALL:
        for suffix, directory, pool in (
            ("profile-export", "profiles", "crypta_profile_export"),
            ("profile-shortterm-interests", "shortterm-interests", "crypta_profile_default"),
        ):
            sandbox.create_scheduler(
                merge_to_bigb_collector.CryptaMergeToBigbCollectorTask,
                kill_timeout=dt(hours=1),
                schedule_interval=dt(minutes=15),
                env=env,
                extra_params={
                    "pool": pool,
                    "fresh_dir": "//home/crypta/{}/profiles/internal-export/bigb-collector/{}/fresh".format(env, directory),
                    "output_dir": "//home/crypta/{}/profiles/internal-export/bigb-collector/{}/output".format(env, directory),
                    "juggler_service_suffix": suffix,
                },
            ).check(
                crit_time=dt(hours=2),
                juggler_service=merge_to_bigb_collector.CryptaMergeToBigbCollectorTask.get_juggler_service_with_suffix(suffix),
            ).set_host("crypta.profile.{}".format(env))


def _add_testing_juggler_checks(juggler):
    juggler = juggler.clone(
        host="crypta.profile.testing",
        ttl=int(dt(hours=32).total_seconds()),
        refresh_time=90,
    )

    _any_events_check(juggler, "copy_to_testing")
    _any_events_check(juggler, "export_profiles").add_nodata_mode(consts.NoDataMode.force_crit)


def _add_production_juggler_checks(juggler):
    juggler = juggler.clone(
        host=COMMON_AGGR_HOSTS.production_host.host,
        refresh_time=90,
    )

    for service, ttl in [
        ("bigb_dump_for_interests", dt(hours=32)),
        ("catalogia_mapping", dt(hours=32)),
        ("consistency_monitoring", dt(hours=32)),
        ("export_profiles", dt(hours=32)),
        ("longterm_interests", dt(hours=32)),
        ("metrics_for_interests", dt(hours=32)),
        ("monitoring", dt(hours=32)),
        ("offline_classification_graph_sample", dt(hours=32)),
        ("precalculate_affinity", dt(hours=32)),
        ("coded_segments", dt(hours=32)),
        ("reqans_for_interests", dt(weeks=1)),
        ("sample_for_lal", dt(hours=32)),
        ("shortterm_interests", dt(hours=32)),
        ("shortterm_interests_upload", dt(hours=2)),
        ("constructor_segments", dt(hours=32)),
        ("validation_sample_building", dt(hours=32)),
    ]:
        _any_events_check(juggler, service).set_ttl(int(ttl.total_seconds())).add_nodata_mode(consts.NoDataMode.force_crit)

    for service, ttl in [
        ("BuildConstructorSegments", dt(hours=28)),
        ("UploadProfilesToLogbroker_crypta_id", dt(hours=28)),
        ("UploadProfilesToLogbroker_yandexuid", dt(hours=28)),
        ("audience_api_upload", dt(hours=28)),
        ("bar_log_parsing", dt(hours=2)),
        ("bb_log_parsing", dt(hours=2)),
        ("geocube_log_parsing", dt(hours=36)),
        ("metrics_log_parsing", dt(hours=2)),
        ("offline_classification_bb_monitoring", dt(hours=32)),
        ("offline_classification_metrica_monitoring", dt(hours=32)),
        ("outdated_coded_segments", dt(hours=32)),
        ("reqans_log_parsing", dt(minutes=80)),
        ("pp_log_parsing", dt(hours=4)),
        ("mobile_bar_log_parsing", dt(hours=2, minutes=30)),
        ("segments_not_in_lab", dt(hours=32)),
    ]:
        juggler.direct(service).set_ttl(int(ttl.total_seconds()))


def _add_output_table_latencies(juggler):
    yt_juggler_gen = juggler.clone(host=COMMON_AGGR_HOSTS.production_host.host)
    yt_registry = yt_config_registry.CryptaYtConfigRegistry(yt_juggler_gen)

    for proxy in YtProxy.Group.offline:
        for path, threshold, escalation in (
            ("profiles/export/daily-export", dt(hours=24), False),
            ("profiles/export/daily-export-cryptaid", dt(hours=24), False),
            ("profiles/export/cryptaid", dt(hours=17), True),
            ("profiles/export/yandexuid", dt(hours=17), True),
        ):
            YtOutputTableLatencyMetric(yt_registry, path, yt_proxy=proxy).add_output_table_latency_alert(threshold).set_phone_escalation(escalation)


def _add_error_dirs_check(juggler):
    yt_juggler_gen = juggler.clone(host=COMMON_AGGR_HOSTS.production_host.host)
    yt_registry = yt_config_registry.CryptaYtConfigRegistry(yt_juggler_gen)

    for proxy in YtProxy.Group.offline:
        for path, threshold, envs in (
            ("profiles/export/lab/direct_exports_tanker_names_errors", DataSize(mb=0), [environment.PRODUCTION]),
        ):
            YtSizeMetric(yt_registry, path, yt_proxy=proxy, envs=envs).add_disk_space_alert(predicate=alert_pb2.GT, threshold=threshold)


def _any_events_check(juggler, service):
    return juggler.any(service).set_child(
        group="tag={} & host={}".format(service, juggler.default_config.host),
        group_type=consts.GroupType.events,
    )


def create_precalculate_tables_scheduler(sandbox, subprogram, kill_timeout, crit_time, schedule_interval, retry_interval=None):
    bundle_name = "profile-precalculate-tables"
    aggr_host = COMMON_AGGR_HOSTS.production_host
    sandbox.create_run_universal_bundle_scheduler(
        bundle_name=bundle_name,
        cmd=["{{cwd}}/crypta-profile-precalculate-tables", "--config", "{{cwd}}/config.yaml", subprogram],
        templates=["config.yaml"],
        env=aggr_host.environment,
        juggler_service=profile_spine_helpers.sandbox_juggler_service(bundle_name, subprogram),
        schedule_interval=schedule_interval,
        kill_timeout=kill_timeout,
        retry_interval=retry_interval,
        semaphore_name=profile_spine_helpers.sandbox_semaphore_name(bundle_name, subprogram, aggr_host.environment),
        sequential_run=True,
    ).check(crit_time=crit_time).set_host(aggr_host.host)
