#!/usr/bin/env python
# -*- coding: utf-8 -*-

import luigi

from crypta.profile.tasks.features.flatten_hits import FlattenHitsBySite
from crypta.profile.tasks.features.merge_hits_by_id import (
    MergeHitsByCryptaid,
    MergeHitsByYandexuid,
)
from crypta.profile.user_vectors import lib as user_vectors
from crypta.profile.utils.config import config
from crypta.profile.utils.loggers import send_to_graphite
from crypta.profile.utils.luigi_utils import (
    BaseYtTask,
    YtDailyRewritableTarget,
)


class CalculateHostIdf(BaseYtTask):
    date = luigi.Parameter()
    data_source = luigi.Parameter()
    id_type = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'

    def requires(self):
        merged_hits_by_id = {
            'yandexuid': MergeHitsByYandexuid(date=self.date, data_source=self.data_source),
            'crypta_id': MergeHitsByCryptaid(date=self.date, data_source=self.data_source),
        }

        return {
            'merged_hits_by_id': merged_hits_by_id[self.id_type],
            'flattened_hits': FlattenHitsBySite(self.date, self.data_source, self.id_type),
        }

    def output(self):
        idf_tables = {
            'crypta_id': {
                'metrics': config.CRYPTA_ID_METRICS_IDF_TABLE,
                'bar': config.CRYPTA_ID_BAR_IDF_TABLE,
            },
            'yandexuid': {
                'metrics': config.YANDEXUID_METRICS_IDF_TABLE,
                'bar': config.YANDEXUID_BAR_IDF_TABLE,
            }
        }
        return YtDailyRewritableTarget(
            idf_tables[self.id_type][self.data_source],
            self.date,
        )

    def run(self):
        with self.yt.Transaction() as transaction:
            self.yql.query(
                user_vectors.calculate_idf_query.format(
                    merged_hits_by_id_table=self.input()['merged_hits_by_id'].table,
                    flattened_hits_table=self.input()['flattened_hits'].table,
                    output_table=self.output().table,
                ),
                transaction=transaction,
                erasure_codec=None,
                compression_codec=None,
            )

            self.yt.set_attribute(self.output().table, 'generate_date', self.date)

            message = 'idf_{source_type}_table'.format(source_type=self.data_source)
            if self.id_type == 'crypta_id':
                message += '_by_crypta_id'

            send_to_graphite(
                message,
                self.yt.row_count(self.output().table),
            )
