#!/usr/bin/env python
# -*- coding: utf-8 -*-

import luigi

from crypta.profile.tasks.features.merge_hits_by_id import (
    MergeHitsByCryptaid,
    MergeHitsByYandexuid,
)
from crypta.profile.user_vectors import lib as user_vectors
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import (
    BaseYtTask,
    YtDailyRewritableTarget,
)


class FlattenHitsBySite(BaseYtTask):
    date = luigi.Parameter()
    data_source = luigi.Parameter()
    id_type = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'

    def requires(self):
        merged_hits_by_id = {
            'yandexuid': MergeHitsByYandexuid(date=self.date, data_source=self.data_source),
            'crypta_id': MergeHitsByCryptaid(date=self.date, data_source=self.data_source),
        }

        return merged_hits_by_id[self.id_type]

    def output(self):
        flattened_hits_tables = {
            'yandexuid': {
                'metrics': config.YANDEXUID_METRICS_FLATTENED_HITS_TABLE,
                'bar': config.YANDEXUID_BAR_FLATTENED_HITS_TABLE,
            },
            'crypta_id': {
                'metrics': config.CRYPTA_ID_METRICS_FLATTENED_HITS_TABLE,
                'bar': config.CRYPTA_ID_BAR_FLATTENED_HITS_TABLE,
            },
        }

        return YtDailyRewritableTarget(
            flattened_hits_tables[self.id_type][self.data_source],
            self.date,
        )

    def run(self):
        with self.yt.Transaction() as transaction:
            self.yql.query(
                user_vectors.flatten_hits_query.format(
                    input_table=self.input().table,
                    output_table=self.output().table,
                    id_type=self.id_type,
                ),
                transaction=transaction,
            )

            self.yt.set_attribute(self.output().table, 'generate_date', self.date)
