#!/usr/bin/env python
# -*- coding: utf-8 -*-

import luigi

from crypta.profile.utils.config import config
from crypta.profile.utils.loggers import send_to_graphite, TimeTracker
from crypta.profile.utils.luigi_utils import YtDailyRewritableTarget, ExternalInput, BaseYtTask


active_pairs_query_template= """
$active_yandexuid = (
    SELECT CAST(yandexuid AS String) AS id
    FROM `{monthly_yandexuid_table}`
);

$active_pairs = (
    SELECT
        CAST(matching.id AS Uint64) AS yandexuid,
        CAST(matching.cryptaId AS Uint64) AS crypta_id
    FROM (
        SELECT id, id_type, cryptaId
        FROM `{vertices_by_id_type_table}`
        WHERE id_type == 'yandexuid'
    ) AS matching
    INNER JOIN $active_yandexuid AS active_yandexuid
    USING (id)
);

INSERT INTO `{active_pairs}` WITH TRUNCATE
SELECT *
FROM $active_pairs;

INSERT INTO `{active_yandexuid}` WITH TRUNCATE
SELECT *
FROM $active_yandexuid;

INSERT INTO `{yandexuid_to_crypta_id_table}` WITH TRUNCATE
SELECT *
FROM $active_pairs
ORDER BY yandexuid;
"""

get_matching_dicts_query_template = """
INSERT INTO `{devid_to_crypta_id_table}` WITH TRUNCATE
SELECT
    matching.id AS devid,
    matching.id AS id,
    matching.id_type AS id_type,
    CAST(matching.cryptaId AS Uint64) AS crypta_id
FROM (
    SELECT id, id_type, cryptaId
    FROM `{vertices_by_id_type_table}`
    WHERE id_type IN AsList('gaid', 'idfa')
) AS matching
INNER JOIN `{app_metrica_month_table}` AS active
USING (id, id_type)
ORDER BY id, id_type;

INSERT INTO `{crypta_id_to_yandexuid_table}` WITH TRUNCATE
SELECT *
FROM `{active_pairs}`
ORDER BY crypta_id;

$active_indevice_pairs = (
    SELECT
        matching.id AS id,
        matching.id_type AS id_type,
        CAST(matching.target_id AS Uint64) AS yandexuid
    FROM (
        SELECT id, id_type, target_id
        FROM `{indevice_yandexuid_by_id_type_table}`
        WHERE id_type IN AsList('puid', 'phone', 'phone_md5', 'email', 'email_md5', 'gaid',
                                'idfa', 'oaid', 'uuid', 'ok_id', 'vk_id', 'dit_id', 'login',
                                'mm_device_id')
    ) AS matching
    INNER JOIN `{active_yandexuid}` AS active_yandexuid
    ON matching.target_id == active_yandexuid.id
);

INSERT INTO `{indevice_yandexuid_table}` WITH TRUNCATE
SELECT *
FROM $active_indevice_pairs
ORDER BY id, id_type;
"""


class GetCryptaIds(BaseYtTask):
    date = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'

    def requires(self):
        return {
            'monthly_yandexuids': ExternalInput(config.MONTHLY_YANDEXUID2VEC),
            'app_metrica_month': ExternalInput(config.APP_METRICA_MONTH),
            'vertices_by_id_type': ExternalInput(config.VERTICES_NO_MULTI_PROFILE_BY_ID_TYPE),
            'indevice_yandexuid_by_id_type': ExternalInput(config.INDEVICE_YANDEXUID_BY_ID_TYPE),
        }

    def output(self):
        return {
            'indevice_yandexuid': YtDailyRewritableTarget(config.INDEVICE_YANDEXUID, date=self.date),
            'cryptaid_yandexuid': YtDailyRewritableTarget(config.CRYPTAID_YANDEXUID_TABLE, date=self.date),
            'devid_cryptaid': YtDailyRewritableTarget(config.DEVID_CRYPTAID_TABLE, date=self.date),
            'yandexuid_cryptaid': YtDailyRewritableTarget(config.YANDEXUID_CRYPTAID_TABLE, date=self.date),
        }

    def run(self):
        with TimeTracker(monitoring_name=self.__class__.__name__):
            with self.yt.Transaction() as transaction,\
                 self.yt.TempTable() as active_pairs,\
                 self.yt.TempTable() as active_yandexuid:

                self.yql.query(
                    query_string=active_pairs_query_template.format(
                        monthly_yandexuid_table=self.input()['monthly_yandexuids'].table,
                        vertices_by_id_type_table=self.input()['vertices_by_id_type'].table,
                        active_pairs=active_pairs,
                        active_yandexuid=active_yandexuid,

                        yandexuid_to_crypta_id_table=self.output()['yandexuid_cryptaid'].table,
                    ),
                    transaction=transaction,
                    erasure_codec=None,  # нужно для аффинитивных сайтов по crypta_id
                )

                self.yql.query(
                    query_string=get_matching_dicts_query_template.format(
                        vertices_by_id_type_table=self.input()['vertices_by_id_type'].table,
                        app_metrica_month_table=self.input()['app_metrica_month'].table,
                        indevice_yandexuid_by_id_type_table=self.input()['indevice_yandexuid_by_id_type'].table,
                        active_pairs=active_pairs,
                        active_yandexuid=active_yandexuid,

                        indevice_yandexuid_table=self.output()['indevice_yandexuid'].table,
                        devid_to_crypta_id_table=self.output()['devid_cryptaid'].table,
                        crypta_id_to_yandexuid_table=self.output()['cryptaid_yandexuid'].table,
                    ),
                    transaction=transaction,
                )

                send_to_graphite(
                    'features.crypta_ids',
                    self.yt.row_count(self.output()['cryptaid_yandexuid'].table)
                )
                send_to_graphite(
                    'features.devid_crypta_ids',
                    self.yt.row_count(self.output()['devid_cryptaid'].table)
                )

                self.yt.set_attribute(self.output()['cryptaid_yandexuid'].table, 'generate_date', self.date)
                self.yt.set_attribute(self.output()['yandexuid_cryptaid'].table, 'generate_date', self.date)
                self.yt.set_attribute(self.output()['devid_cryptaid'].table, 'generate_date', self.date)
                self.yt.set_attribute(self.output()['indevice_yandexuid'].table, 'generate_date', self.date)
