#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

import luigi

from crypta.lib.python import templater
from crypta.profile.lib import date_helpers
from crypta.profile.tasks.features.process_user_events import ProcessUserEvents
from crypta.profile.user_vectors import lib as user_vectors
from crypta.profile.utils.config import config
from crypta.profile.utils.loggers import send_to_graphite
from crypta.profile.utils.luigi_utils import (
    BaseYtTask,
    YtDailyRewritableTarget,
)


MINIMUM_HITS_PER_DAY = 3
HALFLIFE_DAYS = 30


class MergeHitsByYandexuid(BaseYtTask):
    date = luigi.Parameter()
    data_source = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'

    def requires(self):
        return ProcessUserEvents(date=self.date, data_source=self.data_source)

    def output(self):
        merged_hits_by_source_tables = {
            'metrics': config.YANDEXUID_METRICS_MERGED_HITS_TABLE,
            'bar': config.YANDEXUID_BAR_MERGED_HITS_TABLE,
        }

        return YtDailyRewritableTarget(
            merged_hits_by_source_tables[self.data_source],
            self.date,
        )

    def run(self):
        with self.yt.Transaction() as transaction:
            yesterday = date_helpers.get_yesterday(self.date)

            begin_date = date_helpers.get_date_from_past(self.date, days=config.STANDARD_AGGREGATION_PERIOD)
            query = templater.render_template(
                user_vectors.hits_merger_yql,
                vars={
                    'dict_sum': user_vectors.dict_sum_yql,
                    'halflife_days': float(HALFLIFE_DAYS),
                    'minimum_hits_per_day': MINIMUM_HITS_PER_DAY,
                    'minimum_sites': config.MINIMUM_SITES,
                    'maximum_sites': config.MAXIMUM_SITES,
                    'last_active_date': yesterday,
                    'hits_dir': os.path.dirname(self.input()['hits'].table),
                    'begin_date': begin_date,
                    'end_date': yesterday,
                    'merged_hits_table': self.output().table,
                },
            )
            self.yql.query(
                query_string=query,
                transaction=transaction,
                title='YQL Merge {} hits by yandexuid'.format(self.data_source),
            )

            for attribute, value in (
                ('generate_date', self.date),
                ('begin_date', begin_date),
                ('end_date', yesterday),
                ('last_active_date', yesterday),
                ('halflife', HALFLIFE_DAYS),
                ('min_hits', MINIMUM_HITS_PER_DAY),
                ('min_hosts', config.MINIMUM_SITES),
                ('max_hosts', config.MAXIMUM_SITES),
            ):
                self.yt.set_attribute(self.output().table, attribute, value)

            send_to_graphite(
                'merged_{data_source}_hits'.format(data_source=self.data_source),
                self.yt.row_count(self.output().table),
            )


class MergeHitsByCryptaid(BaseYtTask):
    date = luigi.Parameter()
    data_source = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'

    def requires(self):
        return MergeHitsByYandexuid(date=self.date, data_source=self.data_source)

    def output(self):
        merged_hits_by_source_tables = {
            'metrics': config.CRYPTA_ID_METRICS_MERGED_HITS_TABLE,
            'bar': config.CRYPTA_ID_BAR_MERGED_HITS_TABLE,
        }

        return YtDailyRewritableTarget(
            merged_hits_by_source_tables[self.data_source],
            self.date,
        )

    def run(self):
        with self.yt.Transaction() as transaction:
            query = templater.render_template(
                user_vectors.get_cryptaid_merged_hits_yql,
                vars={
                    'dict_sum': user_vectors.dict_sum_yql,
                    'yandexuid_cryptaid_table': config.YANDEXUID_CRYPTAID_TABLE,
                    'merged_hits_by_yandexuid_table': self.input().table,
                    'merged_hits_by_cryptaid_table': self.output().table,
                },
            )
            self.yql.query(
                query_string=query,
                transaction=transaction,
                title='YQL Get merged {} hits by cryptaid'.format(self.data_source),
            )

            self.yt.set_attribute(self.output().table, 'generate_date', self.date)

            send_to_graphite(
                'merged_{data_source}_hits_by_crypta_id'.format(data_source=self.data_source),
                self.yt.row_count(self.output().table),
            )
