#!/usr/bin/env python
# -*- coding: utf-8 -*-

import abc
import os

import luigi
from yt.wrapper import with_context

from crypta.profile.lib import date_helpers

from crypta.profile.utils.config import config
from crypta.profile.utils.loggers import send_to_graphite
from crypta.profile.utils.luigi_utils import BaseYtTask

COUNTRIES_NAMES = {
    '149': 'Belarus',
    '159': 'Kazakhstan',
    '187': 'Ukraine',
    '225': 'Russia',
    '983': 'Turkey',
}


class Monitoring(BaseYtTask):
    date = luigi.Parameter()
    __metaclass__ = abc.ABCMeta
    task_group = 'monitoring'

    @abc.abstractproperty
    def name(self):
        pass

    def __init__(self, date):
        super(Monitoring, self).__init__(date)

        self.yt_folder = os.path.join(config.CRYPTA_MONITORING_FOLDER, self.name, self.date)
        self.yt.config['spec_defaults']['pool'] = config.MONITORING_POOL
        self.yt.create_directory(os.path.dirname(self.yt_folder))

        self.timestamp = date_helpers.from_date_string_to_timestamp(self.date)

    @staticmethod
    def send_to_graphite(name, value, group=None, timestamp=None):
        group = group or 'CRYPTAUP'
        send_to_graphite(name, value, group, timestamp)

    @staticmethod
    def convert_socdem_to_string(socdem):
        if socdem:
            return ''.join(map(str, socdem))
        elif socdem == list():
            return 'unsure'
        else:
            return 'unknown'

    @staticmethod
    def convert_socdem_to_list(socdem):
        if socdem:
            return socdem
        elif socdem == list():
            return ['unsure']
        else:
            return ['unknown']

    def add_yandexuid_country_and_device_type(self, yuid_with_all_table, source_table, destination_table):
        @with_context
        def join_yuid_country_and_device_info_reducer(key, rows, context):
            yandexuid_info_row = None
            for row in rows:
                if context.table_index == 0:
                    yandexuid_info_row = row
                elif context.table_index == 1:
                    row.update(dict.fromkeys(['_country_id', '_device_type']))
                    if yandexuid_info_row:
                        row['_country_id'] = yandexuid_info_row['main_region_country']
                        if yandexuid_info_row['ua_profile']:
                            row['_device_type'] = yandexuid_info_row['ua_profile'].split('|')[1]
                    yield row

        self.yt.run_reduce(
            join_yuid_country_and_device_info_reducer,
            source_table=[yuid_with_all_table, source_table],
            destination_table=destination_table,
            reduce_by='yandexuid',
            spec={'title': 'Add yandexuid country and device type'},
        )

    @staticmethod
    def get_country_name(country_id):
        if country_id:
            return COUNTRIES_NAMES.get(str(country_id), 'others')
        else:
            return 'unknown'

    @staticmethod
    def get_device_type(device_type):
        if device_type in ('desk', 'phone', 'tablet', 'tv'):
            return device_type
        else:
            return 'unknown'
