import os

from crypta.lib.python.juggler.juggler_helpers import report_event_to_juggler
from crypta.profile.tasks.monitoring.__base__ import Monitoring
from crypta.profile.tasks.monitoring.profiles.daily_yandexuid_profiles_monitoring import DailyYandexuidProfilesMonitoring
from crypta.profile.utils import luigi_utils
from crypta.profile.utils.config import config


not_in_lab_segments_query = """
$exported_segments = (
    SELECT segment_id, keyword_id
    FROM `{input_table}`
    GROUP BY segment_id, keyword_id
);

$segments_in_lab = (
    SELECT
        CAST(exportSegmentId AS Uint64) AS segment_id,
        CAST(exportKeywordId AS Uint64) AS keyword_id,
        id
    FROM `{lab_segment_info_table}`
);

$all = (
    SELECT *
    FROM $exported_segments AS exported
    LEFT JOIN $segments_in_lab AS lab
    USING (keyword_id, segment_id)
);


INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    segment_id,
    keyword_id
FROM $all
WHERE id IS NULL;
"""


class NotInLabSegmentsMonitoring(Monitoring):
    name = 'not_in_lab_segments'

    def requires(self):
        return {
            'profiles': DailyYandexuidProfilesMonitoring(self.date),
            'cleaner': luigi_utils.OldNodesByNameCleaner(
                self.date,
                folder=os.path.dirname(self.yt_folder),
                lifetime=5,
            ),
        }

    def output(self):
        return luigi_utils.YtTarget(os.path.join(self.yt_folder, 'segments'), allow_empty=True)

    def run(self):
        self.yql.query(
            not_in_lab_segments_query.format(
                input_table=self.input()['profiles']['segment_count'].table,
                lab_segment_info_table=config.LAB_SEGMENTS_INFO_TABLE,
                output_table=self.output().table,
            ),
        )

        segments_not_in_lab = []
        for row in self.yt.read_table(self.output().table):
            segments_not_in_lab.append('{}:{}'.format(row['keyword_id'], row['segment_id']))

        service_name = 'segments_not_in_lab'
        if len(segments_not_in_lab) != 0:
            message = 'Segments not in lab: {}'.format(', '.join(segments_not_in_lab))
            report_event_to_juggler(
                status='WARN',
                service=service_name,
                host=config.CRYPTA_PROFILE_JUGGLER_HOST,
                description=message,
                logger=self.logger,
            )
        else:
            report_event_to_juggler(
                status='OK',
                service=service_name,
                host=config.CRYPTA_PROFILE_JUGGLER_HOST,
                logger=self.logger,
            )
