import abc
import os

from crypta.lib.python import templater
from crypta.profile.lib import (
    date_helpers,
    pandas_yt,
)
from crypta.profile.lib.socdem_helpers import socdem_config
from crypta.profile.tasks.monitoring.__base__ import Monitoring
from crypta.profile.utils import (
    api,
    luigi_utils,
    segment_storage,
    utils,
)
from crypta.profile.utils.clients import solomon
from crypta.profile.utils.config import config
from crypta.profile.utils.utils import report_ml_metrics_to_solomon


exact_socdem_keywords_mapping = {
    'gender': 174,
    'age_segment': 543,
    'income_5_segment': 614,
}

exact_socdem_segment_ids_mapping = {
    'gender': {
        'm': 0,
        'f': 1,
    },
    'age_segment': {
        '0_17': 0,
        '18_24': 1,
        '25_34': 2,
        '35_44': 3,
        '45_54': 4,
        '55_99': 5,
    },
    'income_5_segment': {
        'A': 0,
        'B1': 1,
        'B2': 2,
        'C1': 3,
        'C2': 4,
    }
}

data_preparation_query_template = """
PRAGMA yson.DisableStrict;

$country_names = AsDict(
    AsTuple(149, 'Belarus'),
    AsTuple(159, 'Kazakhstan'),
    AsTuple(187, 'Ukraine'),
    AsTuple(225, 'Russia'),
    AsTuple(225, 'Russia'),
    AsTuple(983, 'Turkey')
);

$profiles = (
    SELECT
        CAST({{ id_type }} as String) as {{ id_type }},
        Yson::LookupString(exact_socdem, 'gender') ?? 'unsure' AS gender,
        Yson::LookupString(exact_socdem, 'age_segment') ?? 'unsure' AS age_segment,
        Yson::LookupString(exact_socdem, 'income_5_segment') ?? 'unsure' AS income_5_segment,
        ListExtend(
            {% for field_name in dict_fields %}
            ListMap(DictKeys(Yson::ConvertToDict({{ field_name }})), ($x) -> { RETURN AsTuple({{ field_name_to_bb_keyword_id[field_name] }}, $x); }),
            {% endfor %}
            {% for field_name in list_fields %}
            ListMap(Yson::ConvertToUint64List({{ field_name }}), ($x) -> { RETURN AsTuple({{ field_name_to_bb_keyword_id[field_name] }}, CAST($x AS String)); }),
            {% endfor %}
        ) AS segments,
    FROM `{{ profiles_table }}`
);

INSERT INTO `{{ profiles_with_info }}`
WITH TRUNCATE

SELECT
    profiles.{{ id_type }} AS {{ id_type }},
    profiles.gender AS gender,
    profiles.age_segment AS age_segment,
    profiles.income_5_segment AS income_5_segment,
    profiles.segments AS segments,
    CASE
        WHEN info_table.main_region_country IS NOT NULL
        THEN DictLookup($country_names, info_table.main_region_country) ?? 'others'
        ELSE 'unknown'
    END AS country,
    {% if id_type == "yandexuid" %}
    String::SplitToList(info_table.ua_profile, '|')[1] ?? 'unknown' AS device_type,
    {% endif %}
FROM $profiles AS profiles
LEFT JOIN `{{ info_table }}` AS info_table
ON info_table.{{ info_id_type }} == profiles.{{ id_type }};
"""

monitoring_query_template = """
INSERT INTO `{{ socdem_count_table }}`
WITH TRUNCATE

SELECT
    gender,
    age_segment,
    income_5_segment,
    country,
    {% if id_type == "yandexuid" %} device_type, {% endif %}
    COUNT(*) AS `count`,
FROM `{{ profiles_with_info }}`
GROUP BY gender, age_segment, income_5_segment, country {% if id_type == "yandexuid" %}, device_type {% endif %};


INSERT INTO `{{ segment_count_table }}`
WITH TRUNCATE

SELECT
    keyword_id,
    segment_id,
    country,
    {% if id_type == "yandexuid" %} device_type, {% endif %}
    COUNT(*) AS `count`,
FROM (
    SELECT
        {{ id_type }},
        CAST(segment.0 AS Uint64) AS keyword_id,
        CAST(segment.1 AS Uint64) AS segment_id,
        country,
        {% if id_type == "yandexuid" %} device_type, {% endif %}
    FROM `{{ profiles_with_info }}`
    FLATTEN BY segments AS segment
)
GROUP BY keyword_id, segment_id, country {% if id_type == "yandexuid" %}, device_type {% endif %};
"""


class ProfilesMonitoring(Monitoring):

    def output(self):
        return {
            'socdem_count': luigi_utils.YtTarget(os.path.join(self.yt_folder, 'socdem_count')),
            'segment_count': luigi_utils.YtTarget(os.path.join(self.yt_folder, 'segment_count')),
        }

    def run(self):
        with self.yt.Transaction() as transaction, \
             self.yt.TempTable() as profiles_with_info:

            data_preparation_query = templater.render_template(
                data_preparation_query_template,
                {
                    'profiles_table': self.input()['Profiles'].table,
                    'info_table': self.input()['InfoTable'].table,
                    'profiles_with_info': profiles_with_info,
                    'id_type': self.id_type,
                    'dict_fields': segment_storage.DICT_FIELDS,
                    'list_fields': segment_storage.LIST_FIELDS,
                    'field_name_to_bb_keyword_id': utils.field_name_to_bb_keyword_id,
                    'info_id_type': 'crypta_id' if self.id_type == 'crypta_id' else 'id',
                }
            )
            self.yql.query(
                query_string=data_preparation_query,
                transaction=transaction,
            )

            monitoring_query = templater.render_template(
                monitoring_query_template,
                {
                    'socdem_count_table': self.output()['socdem_count'].table,
                    'segment_count_table': self.output()['segment_count'].table,
                    'profiles_with_info': profiles_with_info,
                    'id_type': self.id_type,
                }
            )
            self.yql.query(
                query_string=monitoring_query,
                transaction=transaction,
            )

        # Send data to Graphite
        self.send_segments_count_to_graphite()

        # Send data to Lab
        self.send_count_to_lab()

        # Send data to Solomon
        self.send_segments_count_to_solomon()
        self.send_socdem_count_to_solomon()

    @abc.abstractmethod
    def update_coverage_in_lab(self, export_id, value, timestamp):
        pass

    def send_socdem_count_to_solomon(self):
        df_socdem = pandas_yt.read_into_pandas_dataframe(self.yt, self.output()['socdem_count'].table)
        metrics_to_send = []

        # Send general info about ids_count
        if self.id_type == 'yandexuid':
            groupby_columns = ('country', 'device_type')
        else:
            groupby_columns = 'country'

        for group, count in df_socdem.groupby(groupby_columns)['count'].sum().items():

            if self.id_type == 'yandexuid':
                country, device_type = group
            else:
                country, device_type = group, 'na'

            metrics_to_send.append({
                'labels': {
                    'source': self.solomon_metric_name,
                    'id_type': self.id_type,
                    'country': country,
                    'device_type': device_type,
                    'socdem_type': 'all',
                    'metric': 'count',
                },
                'value': count,
            })

        # Send info about gender, age and income separately
        for socdem_type in socdem_config.EXACT_SOCDEM_FIELDS:
            if self.id_type == 'yandexuid':
                groupby_columns = ('country', 'device_type', socdem_type)
            else:
                groupby_columns = ('country', socdem_type)

            for group, count in df_socdem.groupby(groupby_columns)['count'].sum().items():

                if self.id_type == 'yandexuid':
                    country, device_type, socdem_segment = group
                else:
                    country, socdem_segment = group
                    device_type = 'na'

                metrics_to_send.append({
                    'labels': {
                        'source': self.solomon_metric_name,
                        'id_type': self.id_type,
                        'country': country,
                        'device_type': device_type,
                        'socdem_type': socdem_type,
                        'socdem_segment': socdem_segment,
                        'metric': 'count',
                    },
                    'value': count,
                })

        # Send info about gender and age pairs
        if self.id_type == 'yandexuid':
            groupby_columns = ('country', 'device_type', 'gender', 'age_segment')
        else:
            groupby_columns = ('country', 'gender', 'age_segment')

        for group, count in df_socdem.groupby(groupby_columns)['count'].sum().items():

            if self.id_type == 'yandexuid':
                country, device_type, gender_segment, age_segment = group
            else:
                country, gender_segment, age_segment = group
                device_type = 'na'

            metrics_to_send.append({
                'labels': {
                    'source': self.solomon_metric_name,
                    'id_type': self.id_type,
                    'country': country,
                    'device_type': device_type,
                    'socdem_type': 'gender_age',
                    'gender': gender_segment,
                    'age': age_segment,
                    'metric': 'count',
                },
                'value': count,
            })

        report_ml_metrics_to_solomon(
            service=config.SOLOMON_SOCDEM_DISTRIBUTIONS_SERVICE,
            metrics_to_send=metrics_to_send,
        )

    def send_segments_count_to_graphite(self):
        for row in self.yt.read_table(self.output()['segment_count'].table):
            if self.id_type == 'yandexuid':
                device_type = row['device_type']
            else:
                device_type = 'na'

            name = '{source}.{id_type}.{country}.{device_type}.segments.{keyword_id}.{segment_id}.count'.format(
                source=self.name,
                country=row['country'],
                device_type=device_type,
                keyword_id=row['keyword_id'],
                segment_id=row['segment_id'],
                id_type=self.id_type + 's',
            )

            self.send_to_graphite(
                name=name,
                value=row['count'],
                timestamp=self.timestamp,
            )

    def send_count_to_lab(self):
        # Get Lab segments
        lab_segments = dict()
        for segment in api.get_api().lab.getAllSegments().result():
            for segment_export in segment.exports.exports:
                lab_segments[segment_export.keywordId, segment_export.segmentId] = segment_export.id

        self.send_socdem_count_to_lab(lab_segments)
        self.send_segments_count_to_lab(lab_segments)

    def send_socdem_count_to_lab(self, lab_segments):
        df_socdem = pandas_yt.read_into_pandas_dataframe(self.yt, self.output()['socdem_count'].table)

        # Send socdem coverage to Lab
        for socdem_column in socdem_config.EXACT_SOCDEM_FIELDS:
            keyword_id = exact_socdem_keywords_mapping[socdem_column]
            for socdem_label, count in df_socdem.groupby(socdem_column)['count'].sum().iteritems():
                segment_id = exact_socdem_segment_ids_mapping[socdem_column].get(socdem_label)
                if (keyword_id, segment_id) in lab_segments:
                    segment_export_id = lab_segments[(keyword_id, segment_id)]
                    self.update_coverage_in_lab(
                        export_id=segment_export_id,
                        value=count,
                        timestamp=self.timestamp,
                    )

    def send_segments_count_to_lab(self, lab_segments):
        df_segments = pandas_yt.read_into_pandas_dataframe(self.yt, self.output()['segment_count'].table)

        # Send segments coverage to Lab
        for segment_export_ids, count in df_segments.groupby(('keyword_id', 'segment_id'))['count'].sum().iteritems():
            if segment_export_ids in lab_segments:
                segment_export_id = lab_segments[segment_export_ids]
                self.update_coverage_in_lab(
                    export_id=segment_export_id,
                    value=count,
                    timestamp=self.timestamp,
                )

    def send_segments_count_to_solomon(self):
        solomon_client = solomon.SolomonClient()
        sensors = []
        for row in self.yt.read_table(self.output()['segment_count'].table):
            labels = {
                'keyword_id': str(row['keyword_id']),
                'segment_id': str(row['segment_id']),
                'country': str(row['country']),
            }

            if self.id_type == 'yandexuid':
                labels['device_type'] = str(row['device_type'])
            else:
                labels['device_type'] = 'na'

            sensors.append({
                'labels': labels,
                'ts': date_helpers.from_utc_date_string_to_noon_timestamp(date_helpers.get_yesterday(self.date)),
                'value': row['count'],
            })

        solomon_client.push_sensors(
            project='crypta',
            cluster=config.SOLOMON_CLUSTER,
            service='segments',
            common_labels={'metric': self.solomon_metric_name},
            sensors=sensors,
        )
