#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.utils.config import config
from crypta.profile.tasks.monitoring.__base__ import Monitoring
from crypta.profile.utils.luigi_utils import ExternalInput, ExternalInputDate, YtTarget, OldNodesByNameCleaner


yql_request = """
$has_gender = ($exact_socdem) -> {{RETURN DictContains(Yson::ConvertToStringDict($exact_socdem), 'gender')}};
$has_age_segment = ($exact_socdem) -> {{RETURN DictContains(Yson::ConvertToStringDict($exact_socdem), 'age_segment')}};
$has_gender_and_age_segment = ($exact_socdem) -> {{
    RETURN $has_gender($exact_socdem) AND $has_age_segment($exact_socdem)
}};

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT
    COUNT(devid_profiles.devid) AS total,
    COUNT_IF(os == 'ios') AS ios,
    COUNT_IF(os == 'android') AS android,
    COUNT_IF($has_gender(exact_socdem)) AS has_gender,
    COUNT_IF($has_age_segment(exact_socdem)) AS has_age_segment,
    COUNT_IF($has_gender_and_age_segment(exact_socdem)) AS has_gender_and_age_segment,
    COUNT_IF(os == 'ios' AND $has_gender(exact_socdem)) AS ios_has_gender,
    COUNT_IF(os == 'ios' AND $has_age_segment(exact_socdem)) AS ios_has_age_segment,
    COUNT_IF(os == 'ios' AND $has_gender_and_age_segment(exact_socdem)) AS ios_has_gender_and_age_segment,
    COUNT_IF(os == 'android' AND $has_gender(exact_socdem)) AS android_has_gender,
    COUNT_IF(os == 'android' AND $has_age_segment(exact_socdem)) AS android_has_age_segment,
    COUNT_IF(os == 'android' AND $has_gender_and_age_segment(exact_socdem)) AS android_has_gender_and_age_segment
FROM `{devid_profiles}` AS devid_profiles
INNER JOIN `{dev_info_yt}` AS dev_info_yt
ON (devid_profiles.devid == dev_info_yt.id);
"""


class DevidProfilesSocdemMonitoring(Monitoring):
    name = 'devid_profiles_socdem'

    def requires(self):
        return {
            'dev_info_yt': ExternalInput(
                table=config.APP_METRICA_MONTH,
            ),
            'devid_profiles': ExternalInputDate(
                table=config.DEVID_PROFILES_EXPORT_TABLE,
                date=self.date,
            ),
            'cleaner': OldNodesByNameCleaner(
                self.date,
                folder=os.path.dirname(self.yt_folder),
                lifetime=5,
            ),
        }

    def output(self):
        return YtTarget(os.path.join(self.yt_folder, self.__class__.__name__))

    def run(self):
        query_string = yql_request.format(
            dev_info_yt=self.input()['dev_info_yt'].table,
            devid_profiles=self.input()['devid_profiles'].table,
            output_table=self.output().table,
        )

        self.yql.query(query_string)

        fields = [
            'total',
            'ios',
            'android',
            'has_gender',
            'has_age_segment',
            'has_gender_and_age_segment',
            'ios_has_gender',
            'ios_has_age_segment',
            'ios_has_gender_and_age_segment',
            'android_has_gender',
            'android_has_age_segment',
            'android_has_gender_and_age_segment'
        ]

        formatter = '{source}.{field}'

        for row in self.yt.read_table(self.output().table):
            for field in fields:
                self.send_to_graphite(
                    name=formatter.format(
                        source=self.name,
                        field=field,
                    ),
                    value=row[field],
                    timestamp=self.timestamp,
                )
