#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

from crypta.profile.utils.config import config
from crypta.profile.tasks.monitoring.__base__ import Monitoring
from crypta.profile.utils.luigi_utils import ExternalInput, YtTarget, OldNodesByNameCleaner


yql_request = u"""
$direct_with_crypta_id = 1542;
$user_behavior_targeting = 2;

$filter_non_crypta_targetings = ($targeting_string) -> {{
    $segment_id_and_keyword_id = String::SplitToList($targeting_string, ':');
    $is_crypta_profile_keyword = String::StartsWith($segment_id_and_keyword_id[1], '0@');
    $keyword_id = (
        CASE
            WHEN $is_crypta_profile_keyword
            THEN AsList(
                    CAST(String::SplitToList($segment_id_and_keyword_id[1], '@')[1] AS Uint64),
                    CAST($segment_id_and_keyword_id[0] AS Uint64)
                )
            ELSE NULL
        END
    );
    RETURN $keyword_id
}};

$get_keyword_segment_pairs = ($bs_expression_string) -> {{
    $splitted = String::SplitToList($bs_expression_string, '|()&', False);
    RETURN ListFilter(ListMap($splitted, $filter_non_crypta_targetings), ($elem) -> {{RETURN $elem IS NOT NULL}});
}};

$preprocess_context = (
SELECT
    GoalContextID,
    Bigb::ParseGoalContextProfileProto(GoalContextProfile).GoalContext.Expression AS Expression
FROM `{context_table}`
);

$campaigns_shows_and_clicks = (
    SELECT
        order_info.ClientID AS ClientID,
        order_info.OrderID AS OrderID,
        order_info.ExportID AS ExportID,
        log.groupexportid AS groupexportid,
        context.Expression AS bs_expression,
        COUNT_IF(log.countertype == 1) AS shows,
        COUNT_IF(log.countertype == 2) AS clicks,
        IF(count_if(log.countertype == 1) != 0, Math::Round(100. * count_if(log.countertype == 2) / count_if(log.countertype == 1), -3), 0) AS ctr,
        Math::Round(cast(sum_if(log.costcur, log.countertype == 1) AS Double) / 1e4 / 1.18, -2) AS price, -- расход с учетом откатов, без НДС
        Math::Round(cast(avg_if(log.costcur, log.countertype == 1) AS Double) * 1e3 / 1e4 / 1.18, -2) AS cpm,
        IF(count_if(log.countertype == 2) != 0, Math::Round(cast(sum(log.costcur) AS Double) / 1e4 / 1.18 / count_if(log.countertype == 2), -2), 0) AS cpc,
        is_money, -- оплачен деньгами или сертификатами
        is_deal -- частные сделки
    FROM `{input_table}` AS log
    LEFT JOIN $preprocess_context AS context ON context.GoalContextID = log.phraseid
    JOIN `{order_info_table}` AS order_info ON order_info.OrderID = log.orderid
    WHERE log.placeid == $direct_with_crypta_id
        AND log.fraudbits == 0
        AND log.contexttype == $user_behavior_targeting
    GROUP BY
        order_info.ClientID,
        order_info.OrderID,
        order_info.ExportID,
        log.groupexportid,
        context.Expression,
        (order_info.OrderType != 6) AS is_money,
        order_info.OptionsPrivateDeal AS is_deal
);

$segment_to_campaign_shows_and_clicks = (
    SELECT segment[0] AS keyword_id, segment[1] AS segment_id, shows, clicks, price
    FROM (
        SELECT ClientID, OrderID, ExportID, groupexportid, bs_expression, $get_keyword_segment_pairs(bs_expression) AS segments, shows, clicks, price
        FROM $campaigns_shows_and_clicks
        WHERE ListLength($get_keyword_segment_pairs(bs_expression)) > 0
    )
    FLATTEN BY segments AS segment
);

INSERT INTO `{output_campaigns_table}` WITH TRUNCATE
SELECT *
FROM $campaigns_shows_and_clicks
ORDER BY ClientID, OrderID, groupexportid;

INSERT INTO `{output_table}` WITH TRUNCATE
SELECT keyword_id, segment_id, COUNT(*) AS campaigns_count, SUM(shows) AS sum_shows, SUM(clicks) AS sum_clicks, SUM(price) AS sum_money
FROM $segment_to_campaign_shows_and_clicks
GROUP BY Unwrap(keyword_id) AS keyword_id, Unwrap(segment_id) AS segment_id
ORDER BY sum_shows DESC, sum_clicks DESC
"""

# https://wiki.yandex-team.ru/users/aliho/projects/direct/crypta/#diapazony
bs_keyword_to_crypta_keyword = {
    616: 174,
    617: 543,
    618: 614,
}


class DirectSegmentsUsageMonitoring(Monitoring):
    name = 'direct_segments_usage'

    def requires(self):
        return {
            'context': ExternalInput(
                table='//home/bs/logs/AdsCaesarGoalContextsDump/latest',
            ),
            'order_info': ExternalInput(
                table='//home/yabs/dict/OrderInfo',
            ),
            'bs-chevent-cooked-log': ExternalInput(
                table=os.path.join(config.BS_CHEVENT_COOKED_LOG, self.date)
            ),
            'cleaner': OldNodesByNameCleaner(
                self.date,
                folder=os.path.dirname(self.yt_folder),
                lifetime=5,
            ),
        }

    def output(self):
        return {
            'segments': YtTarget(
                table=os.path.join(self.yt_folder, self.__class__.__name__),
            ),
            'campaigns': YtTarget(
                table=os.path.join(self.yt_folder, 'DirectCampaigns'),
            ),
        }

    def run(self):
        query_string = yql_request.format(
            input_table=self.input()['bs-chevent-cooked-log'].table,
            context_table=self.input()['context'].table,
            order_info_table=self.input()['order_info'].table,
            output_campaigns_table=self.output()['campaigns'].table,
            output_table=self.output()['segments'].table,
        )
        self.yql.query(
            query_string=query_string,
            udf_resource_dict={
                'bigb.so': config.BIGB_UDF_RESOURCE,
            },
        )

        formatter = '{source}.{count_type}.{keyword_id}.{segment_id}'
        for row in self.yt.read_table(self.output()['segments'].table):
            keyword_id = row['keyword_id']
            if keyword_id in bs_keyword_to_crypta_keyword:
                keyword_id = bs_keyword_to_crypta_keyword[keyword_id]
            for count_type in 'campaigns_count', 'sum_shows', 'sum_clicks', 'sum_money':
                self.send_to_graphite(
                    name=formatter.format(
                        source=self.name,
                        count_type=count_type,
                        keyword_id=keyword_id,
                        segment_id=row['segment_id'],
                    ),
                    value=row[count_type],
                    timestamp=self.timestamp,
                )
