#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

import luigi

from crypta.lib.python.juggler.juggler_helpers import report_event_to_juggler
from crypta.profile.lib import date_helpers
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import (
    BaseYtTask,
    YtNodeAttributeTarget,
)


class SegmentFreshnessMonitoring(BaseYtTask):
    date = luigi.Parameter()
    segment_type = luigi.Parameter()

    task_group = 'monitoring'
    name = 'segment_freshness_monitoring'

    def get_tables_to_check(self):
        tables_to_check = self.yt.list(
            os.path.join(config.PROFILES_SEGMENT_PARTS_YT_DIRECTORY, self.segment_type),
            absolute=True,
        )

        return tables_to_check

    def report_outdated_tables(self, outdated_tables):
        service_name = 'outdated_coded_segments'

        self.logger.info('Outdated tables: {}'.format(', '.join(outdated_tables)))

        if self.segment_type == config.TRAINABLE_SEGMENTS:
            juggler_host = config.CRYPTA_ML_JUGGLER_HOST
        else:
            juggler_host = config.CRYPTA_PROFILE_JUGGLER_HOST

        if outdated_tables:
            message = 'Outdated tables: {tables}'.format(tables=', '.join(outdated_tables))
            report_event_to_juggler(
                status='WARN',
                service=service_name,
                host=juggler_host,
                description=message,
                logger=self.logger,
            )
        else:
            report_event_to_juggler(
                status='OK',
                service=service_name,
                host=juggler_host,
                logger=self.logger,
            )

    def run(self):
        tables_to_check = self.get_tables_to_check()
        fresh_date = date_helpers.get_date_from_past(self.date, 2)

        outdated_tables = []
        for table in tables_to_check:
            generate_date = self.yt.get_attribute(table, 'generate_date', None)

            if generate_date is not None:
                if generate_date < fresh_date:
                    outdated_tables.append(os.path.basename(table))
            else:
                self.logger.info('No generate_date for {}'.format(table))
                outdated_tables.append(os.path.basename(table))

        self.report_outdated_tables(outdated_tables)

        self.yt.set_attribute(
            self.output().path,
            self.__class__.__name__,
            self.date,
        )

    def output(self):
        return YtNodeAttributeTarget(
            path=os.path.join(config.PROFILES_SEGMENT_PARTS_YT_DIRECTORY, self.segment_type),
            attribute_name=self.__class__.__name__,
            attribute_value=self.date,
        )
