#!/usr/bin/env python
# -*- coding: utf-8 -*-

from datetime import timedelta
import os

from crypta.lib.python.yt import yt_helpers
from crypta.profile.lib import pandas_yt
from crypta.profile.lib.socdem_helpers import socdem_config
from crypta.profile.tasks.monitoring.__base__ import Monitoring
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import ExternalInputDate, YtTarget
from crypta.profile.utils.utils import report_ml_metrics_to_solomon


sample_monitoring_query = """
PRAGMA yson.DisableStrict;

$sample = (
    SELECT
        yandexuid,
        gender ?? 'unknown' AS gender,
        age_segment ?? 'unknown' AS age_segment,
        income_segment ?? 'unknown' AS income_segment,
        'socdem_labels_for_learning' AS monitoring_type,
    FROM `{socdem_labels_for_learning_table}`
    UNION ALL
    SELECT
        yandexuid,
        gender ?? 'unknown' AS gender,
        age_segment ?? 'unknown' AS age_segment,
        income_segment ?? 'unknown' AS income_segment,
        'socdem_training_sample' AS monitoring_type,
    FROM `{socdem_training_sample_table}`
    UNION ALL
    SELECT
        yandexuid,
        gender ?? 'unknown' AS gender,
        age_segment ?? 'unknown' AS age_segment,
        income_segment ?? 'unknown' AS income_segment,
        'socdem_labels' AS monitoring_type,
    FROM `{socdem_labels_table}`
);

$country_names = AsDict(
    AsTuple(149, 'Belarus'),
    AsTuple(159, 'Kazakhstan'),
    AsTuple(187, 'Ukraine'),
    AsTuple(225, 'Russia'),
    AsTuple(225, 'Russia'),
    AsTuple(983, 'Turkey')
);

$sample_with_info = (
    SELECT
        sample.yandexuid AS yandexuid,
        sample.gender AS gender,
        sample.age_segment AS age_segment,
        sample.income_segment AS income_segment,
        sample.monitoring_type AS monitoring_type,
        CASE
            WHEN yandexuid_info.main_region_country IS NOT NULL
            THEN DictLookup($country_names, yandexuid_info.main_region_country) ?? 'others'
            ELSE 'unknown'
        END AS country,
        String::SplitToList(yandexuid_info.ua_profile, '|')[1] ?? 'unknown' AS device_type,
    FROM $sample AS sample
    LEFT JOIN `{yuid_with_all_table}` AS yandexuid_info
    USING (yandexuid)
);

INSERT INTO `{sample_stat_table}`
WITH TRUNCATE

SELECT
    monitoring_type,
    gender,
    age_segment,
    income_segment,
    country,
    device_type,
    COUNT(*) AS `count`,
FROM $sample_with_info
GROUP BY monitoring_type, gender, age_segment, income_segment, country, device_type;
"""


class SocdemSampleMonitoring(Monitoring):
    name = 'socdem_sample'

    def requires(self):
        return {
            'socdem_training_sample': ExternalInputDate(
                config.SOCDEM_TRAINING_SAMPLE_TABLE,
                self.date
            ),
            'socdem_labels_for_learning': ExternalInputDate(
                config.SOCDEM_LABELS_FOR_LEARNING_TABLE,
                self.date,
            ),
            'socdem_labels': ExternalInputDate(
                config.SOCDEM_LABELS_TABLE,
                self.date,
            ),
        }

    def output(self):
        return YtTarget(os.path.join(self.yt_folder, 'sample_stat_count'))

    def run(self):
        with self.yt.Transaction() as transaction:
            self.yql.query(
                sample_monitoring_query.format(
                    socdem_labels_for_learning_table=self.input()['socdem_labels_for_learning'].table,
                    socdem_training_sample_table=self.input()['socdem_training_sample'].table,
                    socdem_labels_table=self.input()['socdem_labels'].table,
                    yuid_with_all_table=config.YUID_WITH_ALL_BY_YANDEXUID_TABLE,
                    sample_stat_table=self.output().table,
                ),
                transaction=transaction,
            )

            yt_helpers.set_ttl(
                table=self.output().table,
                ttl_timedelta=timedelta(days=config.MONITORING_TABLES_TTL_DAYS),
                yt_client=self.yt,
            )

        self.send_socdem_count_to_solomon()

    def send_socdem_count_to_solomon(self):
        df_socdem = pandas_yt.read_into_pandas_dataframe(self.yt, self.output().table)
        metrics_to_send = []

        # send general info about yandexuids_count
        groupby_columns = ('country', 'device_type', 'monitoring_type')
        for group, count in df_socdem.groupby(groupby_columns)['count'].sum().iteritems():
            country, device_type, monitoring_name = group
            metrics_to_send.append({
                'labels': {
                    'source': monitoring_name,
                    'country': country,
                    'device_type': device_type,
                    'socdem_type': 'all',
                    'metric': 'count',
                },
                'value': count,
            })

        # send info about gender, age and income separately
        for socdem_type in socdem_config.SOCDEM_SEGMENT_TYPES:
            groupby_columns = ('country', 'device_type', 'monitoring_type', socdem_type)
            for group, count in df_socdem.groupby(groupby_columns)['count'].sum().iteritems():
                country, device_type, monitoring_name, socdem_segment = group
                metrics_to_send.append({
                    'labels': {
                        'source': monitoring_name,
                        'country': country,
                        'device_type': device_type,
                        'socdem_type': socdem_type,
                        'socdem_segment': socdem_segment,
                        'metric': 'count',
                    },
                    'value': count,
                })

        # Send info about gender and age pairs
        groupby_columns = ('country', 'device_type', 'monitoring_type', 'gender', 'age_segment')
        for group, count in df_socdem.groupby(groupby_columns)['count'].sum().iteritems():
            country, device_type, monitoring_name, gender_segment, age_segment = group
            metrics_to_send.append({
                'labels': {
                    'source': monitoring_name,
                    'country': country,
                    'device_type': device_type,
                    'socdem_type': 'gender_age',
                    'gender': gender_segment,
                    'age': age_segment,
                    'metric': 'count',
                },
                'value': count,
            })

        report_ml_metrics_to_solomon(
            service=socdem_config.SOLOMON_SERVICE,
            metrics_to_send=metrics_to_send,
        )
