#!/usr/bin/env python
# -*- coding: utf-8 -*-

import itertools
import os

import luigi

from crypta.lib.python.yt import yt_helpers
from crypta.profile.lib import pandas_yt
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import BaseYtTask, YtNodeAttributeTarget
from crypta.profile.utils.utils import report_ml_metrics_to_solomon

from crypta.profile.tasks.monitoring.validation_by_sources.validate_gender_age_profiles import ValidateGenderAgeProfilesBySources
from crypta.profile.tasks.monitoring.validation_by_sources.validate_income_profiles import ValidateIncomeProfilesBySources


REGIONS = ('russia', 'moscow')


class SendValidationResults(BaseYtTask):
    date = luigi.Parameter()
    juggler_host = config.CRYPTA_ML_JUGGLER_HOST
    task_group = 'validate_socdem_profiles'
    sources = dict()

    def requires(self):
        return {
            'gender_age': ValidateGenderAgeProfilesBySources(self.date),
            'income': ValidateIncomeProfilesBySources(self.date),
        }

    def output(self):
        outputs = {}

        for input_name, input_task in itertools.chain(self.input()['gender_age'].items(), self.input()['income'].items()):
            outputs['{}_{}'.format(input_name, 'write_stats_to_yt')] = YtNodeAttributeTarget(
                path=input_task.path,
                attribute_name='write_stats_to_yt',
                attribute_value=True,
            )

        return outputs

    def get_socdem_stats(self, date):
        import pandas as pd

        socdem_stats = {
            'age': [],
            'gender': [],
            'income': [],
        }

        for source, region in itertools.product(self.sources, REGIONS):
            for socdem_type in self.sources[source]:
                socdem_info_table = os.path.join(
                    config.SOCDEM_VALIDATION_DIR,
                    source,
                    date,
                    region,
                    '{}_stats'.format(socdem_type),
                )

                df = pandas_yt.read_into_pandas_dataframe(
                    self.yt,
                    self.yt.TablePath(
                        socdem_info_table,
                        columns=(socdem_type, 'validation_count', 'crypta_count', 'intersection_count'),
                    ),
                )
                df['region'] = region
                df['source'] = source

                socdem_stats[socdem_type].append(df)

        for socdem_type in socdem_stats:
            socdem_stats[socdem_type] = pd.concat(socdem_stats[socdem_type])

        return socdem_stats

    def write_validation_result_to_yt(self, df, socdem_type):
        if socdem_type == 'gender':
            df = df[df['source'] != 'socialdb']

        data = []
        for (region, socdem_segment), counts in df.groupby(['region', socdem_type]).sum().iterrows():
            intersection_count = counts['intersection_count']
            crypta_count = counts['crypta_count']
            precision = round(100.0 * intersection_count / crypta_count, 1)
            data.append({
                'fielddate': self.date,
                'table_type': 'profiles_{}'.format(region),
                'socdem': socdem_type,
                'segment': socdem_segment,
                'accuracy': precision,
            })
        yt_helpers.write_stats_to_yt(
            yt_client=self.yt,
            table_path=config.DATALENS_SOCDEM_VALIDATION_TABLE,
            data_to_write=data,
            schema={
                'fielddate': 'string',
                'segment': 'string',
                'socdem': 'string',
                'table_type': 'string',
                'accuracy': 'double',
            },
        )

    @staticmethod
    def send_validation_result_to_solomon(df, socdem_type):
        gender_convertion_dict = {'f': 'female', 'm': 'male', 'total': 'total'}

        metrics_to_send = []
        for row in df.iterrows():
            for count_type in ('validation_count', 'crypta_count', 'intersection_count'):
                metrics_to_send.append({
                    'labels': {
                        'region': row[1]['region'],
                        'source': row[1]['source'],
                        'socdem': socdem_type,
                        'segment': gender_convertion_dict[row[1][socdem_type]] if socdem_type == 'gender' else row[1][socdem_type],
                        'metric': count_type.replace('validation', row[1]['source']),
                    },
                    'value': row[1][count_type],
                })

        report_ml_metrics_to_solomon(
            service=config.SOLOMON_SOCDEM_VALIDATION_SERVICE,
            metrics_to_send=metrics_to_send,
        )

    def run(self):
        self.sources = {
            input_name: task_name.split('_') for task_name, task in self.input().items() for input_name in task
        }

        socdem_stats = self.get_socdem_stats(self.date)
        for socdem_type in ('gender', 'age', 'income'):
            self.write_validation_result_to_yt(socdem_stats[socdem_type], socdem_type)
            if socdem_type != 'income':
                self.send_validation_result_to_solomon(socdem_stats[socdem_type], socdem_type)

        for source in self.sources:
            self.yt.set_attribute(
                os.path.join(config.SOCDEM_VALIDATION_DIR, source, self.date),
                'write_stats_to_yt',
                True,
            )
