#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os

import luigi

from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import (
    BaseYtTask,
    YtNodeAttributeTarget,
    OldNodesByNameCleaner,
    ExternalInput,
)
from crypta.profile.tasks.monitoring.validation_by_sources import utils
from crypta.profile.tasks.monitoring.validation_by_sources.make_samples import MakeSocdemValidationSamples


make_profiles_validation_samples_query_template = """
$validation_yandexuid = (
    SELECT
        indevice_yandexuid.yandexuid AS yandexuid,
        validation.validation_gender AS validation_gender,
        validation.validation_age AS validation_age
    FROM `{validation_table}` AS validation
    INNER JOIN `{indevice_yandexuid_table}` AS indevice_yandexuid
    USING (id, id_type)
);

$exact_socdem = ($exact_socdem) -> {{RETURN Yson::ConvertToStringDict($exact_socdem)}};

$profiles_validation = (
    SELECT
        validation.yandexuid AS yandexuid,
        validation.validation_age AS validation_age,
        validation.validation_gender AS validation_gender,
        $exact_socdem(profiles.exact_socdem)['age_segment'] AS predicted_age,
        $exact_socdem(profiles.exact_socdem)['gender'] AS predicted_gender
    FROM $validation_yandexuid AS validation
    INNER JOIN `{profiles_table}` AS profiles
    USING (yandexuid)
    WHERE profiles.update_time >= $max_update_time
);

INSERT INTO `{russia_output}` WITH TRUNCATE
SELECT
    validation.yandexuid AS yandexuid,
    validation.validation_age AS validation_age,
    validation.validation_gender AS validation_gender,
    validation.predicted_age AS predicted_age,
    validation.predicted_gender AS predicted_gender
FROM $profiles_validation AS validation
INNER JOIN `{yuid_with_all_table}` AS yuid_with_all
USING (yandexuid)
WHERE yuid_with_all.main_region_country == 225
ORDER BY yandexuid;

INSERT INTO `{moscow_output}` WITH TRUNCATE
SELECT
    validation.yandexuid AS yandexuid,
    validation.validation_age AS validation_age,
    validation.validation_gender AS validation_gender,
    validation.predicted_age AS predicted_age,
    validation.predicted_gender AS predicted_gender
FROM $profiles_validation AS validation
INNER JOIN `{yuid_with_all_table}` AS yuid_with_all
USING(yandexuid)
WHERE yuid_with_all.main_region_city == 213
ORDER BY yandexuid;
"""


class ValidateGenderAgeProfilesBySources(BaseYtTask):
    date = luigi.Parameter()
    juggler_host = config.CRYPTA_ML_JUGGLER_HOST
    task_group = 'validate_socdem_profiles'

    def requires(self):
        return {
            'CryptaProfiles': ExternalInput(
                os.path.join(
                    config.YANDEXUID_PROFILES_EXPORT_DIRECTORY,
                    self.date,
                ),
            ),
            'ValidationSamples': MakeSocdemValidationSamples(self.date),
            'CleanerPassport': OldNodesByNameCleaner(
                date=self.date,
                folder=os.path.join(config.SOCDEM_VALIDATION_DIR, 'passport'),
                lifetime=14,
            ),
            'CleanerPeoplesearchVK': OldNodesByNameCleaner(
                date=self.date,
                folder=os.path.join(config.SOCDEM_VALIDATION_DIR, 'peoplesearch_vk'),
                lifetime=14,
            ),
            'CleanerSocialdb': OldNodesByNameCleaner(
                date=self.date,
                folder=os.path.join(config.SOCDEM_VALIDATION_DIR, 'socialdb'),
                lifetime=14,
            ),
            'CleanerCustomInput': OldNodesByNameCleaner(
                date=self.date,
                folder=os.path.join(config.SOCDEM_VALIDATION_DIR, 'custom_input'),
                lifetime=14,
            ),
            'CleanerCustomOutput': OldNodesByNameCleaner(
                date=self.date,
                folder=os.path.join(config.SOCDEM_VALIDATION_DIR, 'custom_output'),
                lifetime=14,
            ),
        }

    def output(self):
        return {
            'passport': YtNodeAttributeTarget(
                path=os.path.join(config.SOCDEM_VALIDATION_DIR, 'passport', self.date),
                attribute_name='processed',
                attribute_value=True,
            ),
            'peoplesearch_vk': YtNodeAttributeTarget(
                path=os.path.join(config.SOCDEM_VALIDATION_DIR, 'peoplesearch_vk', self.date),
                attribute_name='processed',
                attribute_value=True,
            ),
            'socialdb': YtNodeAttributeTarget(
                path=os.path.join(config.SOCDEM_VALIDATION_DIR, 'socialdb', self.date),
                attribute_name='processed',
                attribute_value=True,
            ),
        }

    def run(self):
        with self.yt.Transaction() as transaction:
            for source in self.input()['ValidationSamples']:
                self.yql.query(
                    query_string='{}\n{}'.format(
                        utils.calculate_max_update_time.format(
                            profiles_table=self.input()['CryptaProfiles'].table,
                        ),
                        make_profiles_validation_samples_query_template.format(
                            validation_table=os.path.join(config.SOCDEM_VALIDATION_DIR, source, self.date, 'sample'),
                            profiles_table=self.input()['CryptaProfiles'].table,
                            calculate_max_update_time=utils.calculate_max_update_time,
                            indevice_yandexuid_table=config.INDEVICE_YANDEXUID,
                            yuid_with_all_table=config.YUID_WITH_ALL_BY_YANDEXUID_TABLE,
                            russia_output=os.path.join(
                                config.SOCDEM_VALIDATION_DIR, source, self.date, 'russia', 'sample',
                            ),
                            moscow_output=os.path.join(
                                config.SOCDEM_VALIDATION_DIR, source, self.date, 'moscow', 'sample',
                            ),
                        ),
                    ),
                    transaction=transaction,
                )

                for region_type in ('russia', 'moscow'):
                    for segment_type in ('age', 'gender'):
                        self.yql.query(
                            query_string=utils.calculate_stats_query_template.format(
                                input_table=os.path.join(
                                    config.SOCDEM_VALIDATION_DIR, source,
                                    self.date, region_type, 'sample',
                                ),
                                segment_type=segment_type,
                                output_table=os.path.join(
                                    config.SOCDEM_VALIDATION_DIR, source, self.date, region_type,
                                    '{}_stats'.format(segment_type),
                                ),
                            ),
                            transaction=transaction,
                        )

                self.yt.set_attribute(
                    os.path.join(config.SOCDEM_VALIDATION_DIR, source, self.date),
                    'processed',
                    True,
                )
