#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
import os

import luigi

from crypta.lib.python import templater
from crypta.lib.python.yt import yt_helpers
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import (
    BaseYtTask,
    YtNodeAttributeTarget,
    ExternalInput,
)
from crypta.profile.tasks.monitoring.validation_by_sources import utils


make_profiles_validation_samples_query_template = """
{% if source == 'delta_credit' %}
$percentile_A = {{percentile_A}};
$percentile_B1 = {{percentile_B1}};
$percentile_B2 = {{percentile_B2}};
$percentile_C1 = {{percentile_C1}};

$income_segment = ($salary) -> {
    RETURN CASE
        WHEN $salary <= $percentile_A THEN 'A'
        WHEN $salary > $percentile_A AND $salary <= $percentile_B1 THEN 'B1'
        WHEN $salary > $percentile_B1 AND $salary <= $percentile_B2 THEN 'B2'
        WHEN $salary > $percentile_B2 AND $salary <= $percentile_C1 THEN 'C1'
        ELSE 'C2'
    END
};
{% endif %}

$max_update_time = (SELECT MAX(update_time) FROM `{{profiles_table}}`);

$income_validation = (
    SELECT
        {% if source == 'delta_credit' %}
        $income_segment(validation.salary) AS validation_income,
        {% else %}
        income_5_segment AS validation_income,
        {% endif %}
        indevice_yandexuid.yandexuid AS yandexuid,
        Yson::ConvertToStringDict(profiles.exact_socdem)['income_5_segment'] AS predicted_income,
    FROM `{{validation_table}}` AS validation
    INNER JOIN `{{indevice_yandexuid_table}}` AS indevice_yandexuid
    ON validation.id == indevice_yandexuid.id AND validation.id_type == indevice_yandexuid.id_type
    INNER JOIN `{{profiles_table}}` AS profiles
    ON indevice_yandexuid.yandexuid == profiles.yandexuid
    WHERE profiles.update_time >= $max_update_time
);

{% for output_table, region in sample_parameters %}

INSERT INTO `{{output_table}}`
WITH TRUNCATE

SELECT validation.*
FROM $income_validation AS validation
INNER JOIN `{{yuid_with_all_table}}` AS yuid_with_all
USING (yandexuid)
{% if region == 'Russia' %}
WHERE yuid_with_all.main_region_country == 225
{% elif region == 'Moscow' %}
WHERE yuid_with_all.main_region_city == 213
{% endif %}
ORDER BY yandexuid;

{% endfor %}
"""


class ValidateIncomeProfilesBySources(BaseYtTask):
    date = luigi.Parameter()
    juggler_host = config.CRYPTA_ML_JUGGLER_HOST
    task_group = 'validate_socdem_profiles'

    def requires(self):
        return {
            'crypta_profiles': ExternalInput(
                os.path.join(
                    config.YANDEXUID_PROFILES_EXPORT_DIRECTORY,
                    self.date,
                ),
            ),
            'delta_credit': ExternalInput(
                os.path.join(
                    config.SCORING_FOLDER,
                    'delta_credit_2019-02-01_validation',
                ),
            ),
            'beeline': ExternalInput(os.path.join(config.TELECOMS_FOLDER, 'beeline_validation')),
        }

    def output(self):
        return {
            'delta_credit': YtNodeAttributeTarget(
                path=os.path.join(config.SOCDEM_VALIDATION_DIR, 'delta_credit', self.date),
                attribute_name='processed',
                attribute_value=True,
            ),
            'beeline': YtNodeAttributeTarget(
                path=os.path.join(config.SOCDEM_VALIDATION_DIR, 'beeline', self.date),
                attribute_name='processed',
                attribute_value=True,
            ),
        }

    def run(self):
        with self.yt.Transaction() as transaction:
            income_thresholds = next(self.yt.read_table(config.INCOME_THRESHOLDS_TABLE))

            for source in self.output():
                make_profiles_validation_samples_query = templater.render_template(
                    template_text=make_profiles_validation_samples_query_template,
                    vars={
                        'source': source,
                        'percentile_A': income_thresholds['A_upper_threshold'],
                        'percentile_B1': income_thresholds['B1_upper_threshold'],
                        'percentile_B2': income_thresholds['B2_upper_threshold'],
                        'percentile_C1': income_thresholds['C1_upper_threshold'],
                        'profiles_table': self.input()['crypta_profiles'].table,
                        'yuid_with_all_table': config.YUID_WITH_ALL_BY_YANDEXUID_TABLE,
                        'validation_table': self.input()[source].table,
                        'indevice_yandexuid_table': config.INDEVICE_YANDEXUID,
                        'sample_parameters': (
                            (
                                os.path.join(self.output()[source].path, 'russia', 'sample'),
                                'Russia',
                            ),
                            (
                                os.path.join(self.output()[source].path, 'moscow', 'sample'),
                                'Moscow',
                            ),
                        ),
                    },
                )

                self.yql.query(
                    query_string=make_profiles_validation_samples_query,
                    transaction=transaction,
                    title='Join {} validation sample with active profiles'.format(source),
                )

                for region_type in ('russia', 'moscow'):
                    self.yql.query(
                        query_string=utils.calculate_stats_query_template.format(
                            input_table=os.path.join(
                                self.output()[source].path, region_type, 'sample',
                            ),
                            segment_type='income',
                            output_table=os.path.join(
                                self.output()[source].path, region_type, 'income_stats',
                            ),
                        ),
                        transaction=transaction,
                    )

                self.yt.set_attribute(
                    self.output()[source].path,
                    'processed',
                    True,
                )

                yt_helpers.set_ttl(
                    self.output()[source].path,
                    ttl_timedelta=datetime.timedelta(days=14),
                    yt_client=self.yt,
                )
