import json
import tempfile

from crypta.profile.utils.config import config
from crypta.profile.utils.utils import download_file_from_sandbox


BIGB_AB_EXPERIMENTS_PRODUCTION_CONFIG = 'BIGB_AB_EXPERIMENTS_PRODUCTION_CONFIG'
BS_YETI_LIGHT_CONFIGS = 'BS_YETI_LIGHT_CONFIGS'


split_exp_default_query = '''
PRAGMA yt.DataSizePerJob('10m');

PRAGMA File('bigb.so', 'https://proxy.sandbox.yandex-team.ru/last/BIGB_UDF?attrs={{"released":"stable"}}');
PRAGMA udf('bigb.so');

PRAGMA File("select_type_active", "https://proxy.sandbox.yandex-team.ru/last/BS_YETI_LIGHT_CONFIGS/select_type_active.json");
PRAGMA File("ab_experiments_config", "https://proxy.sandbox.yandex-team.ru/last/LONG_AB_EXPERIMENTS_CONFIG");

$exps = Bigb::LoadExperimentSystemWithAb(FilePath('select_type_active'), FilePath('ab_experiments_config'), true);

$timestamp = {timestamp};
$exp = 20{test_id};

$GetActiveExperiments = ($crypta_id) -> {{
    RETURN Bigb::GetActiveExperimentsForIds($exps, AsStruct(1 AS UniqId, $crypta_id as CryptaId2), $timestamp);
}};

$GetLabel = ($crypta_id) -> {{
    $exp_ids = ListMap(
        $GetActiveExperiments($crypta_id),
        ($each) -> {{
            RETURN $each.ExperimentID;
        }}
    );
    RETURN IF($exp IN $exp_ids, 'exp', 'default');
}};

$labeled = (
    SELECT
        input_table.*,
        $GetLabel(matching.crypta_id) AS label
    FROM `{input_table}` AS input_table
    LEFT JOIN ANY `{matching_to_cryptaid}` AS matching
    USING({matching_join_key})
);

INSERT INTO `{output_default}`
WITH TRUNCATE

SELECT labeled.* WITHOUT label
FROM $labeled AS labeled
WHERE label == 'default';

INSERT INTO `{output_exp}`
WITH TRUNCATE

SELECT labeled.* WITHOUT label
FROM $labeled AS labeled
WHERE label == 'exp';
'''


def get_config_resource(
    resource_type,
    path=None,
    read_as_json=False,
    environment=config.environment,
):
    if environment != 'production':
        return
    with tempfile.NamedTemporaryFile() as resource_file_to_save:
        download_file_from_sandbox(
            resource_type=resource_type,
            released='stable',
            path=path,
            path_to_save=resource_file_to_save.name,
        )
        with open(resource_file_to_save.name, 'r') as resource_file_to_load:
            if read_as_json:
                return json.load(resource_file_to_load)

            return resource_file_to_load.read()


def get_select_type(
    environment=config.environment,
):
    return get_config_resource(
        resource_type=BS_YETI_LIGHT_CONFIGS,
        path='select_type_active.json',
        read_as_json=True,
        environment=environment,
    )


def get_ab_config(
    environment=config.environment,
):
    return get_config_resource(
        resource_type=BIGB_AB_EXPERIMENTS_PRODUCTION_CONFIG,
        environment=environment,
    )


def get_exp_system(
    select_type,
    ab_config,
    environment=config.environment,
):
    if environment == 'production':
        from ads.bsyeti.libs.py_experiments import TExpSystem
        return TExpSystem(
            select_type=select_type,
            ab_config=ab_config,
            long_config=True,
        )


def get_ab_group(
    test_id,
    timestamp,
    exp_system,
    crypta_id,
    environment=config.environment,
):
    if environment == 'production':
        exp_info_for_crypta_id = exp_system.get_experiments(
            uniq_id=1,
            crypta_id2=crypta_id,
            show_time=timestamp,
        )
        exp_ids_for_crypta_id = map(
            lambda exp_info: exp_info[0],
            exp_info_for_crypta_id
        )
        return int(int('20' + str(test_id)) in exp_ids_for_crypta_id)

    return 0


class CryptaIdExperiment:
    def __init__(
        self,
        yt,
        yql,
        test_id,
        timestamp,
    ):

        self.yt = yt
        self.yql = yql
        self.test_id = test_id
        self.timestamp = timestamp

    def exp_default_map(
        self,
        input_table_path,
        exp_table_path,
        default_table_path,
        is_mobile=False,
        execute=False,
    ):
        query = split_exp_default_query.format(
            test_id=self.test_id,
            timestamp=self.timestamp,
            input_table=input_table_path,
            matching_to_cryptaid=config.DEVID_CRYPTAID_TABLE if is_mobile else config.YANDEXUID_CRYPTAID_TABLE,
            matching_join_key='id, id_type' if is_mobile else 'yandexuid',
            output_default=default_table_path,
            output_exp=exp_table_path,
        )

        if execute:
            self.yql.execute(
                query=query,
            )
        else:
            return query

    def devid_and_yandexuid_exp_default_map(
        self,
        yandexuid_input_table_path,
        yandexuid_exp_table_path,
        yandexuid_default_table_path,
        devid_input_table_path,
        devid_exp_table_path,
        devid_default_table_path,
    ):
        self.yql.execute(
            query='{yandexuid_query}\n{devid_query}'.format(
                yandexuid_query=self.exp_default_map(
                    yandexuid_input_table_path,
                    yandexuid_exp_table_path,
                    yandexuid_default_table_path,
                    is_mobile=False,
                ),
                devid_query=self.exp_default_map(
                    devid_input_table_path,
                    devid_exp_table_path,
                    devid_default_table_path,
                    is_mobile=True,
                )
            ),
        )
