#!/usr/bin/env python
# -*- coding: utf-8 -*-

import tempfile
import tvmauth

from crypta.lib.python.audience import client
from crypta.lib.python.tvm.helpers import get_tvm_headers

from crypta.profile.utils.config import config
from crypta.profile.utils.config.secrets import get_secrets
from crypta.profile.utils.loggers import get_logger
from crypta.profile.utils.yt_utils import get_yt_client


AUDIENCE_RETRIES = 3
AUDIENCE_API_URL = 'https://api-audience.yandex.ru'
AUDIENCE_API_PORT = 443
AUDIENCE_MIN_SEGMENT_SIZE = 1000


class AudienceClient(client.PrivateApiAudienceClient):
    """https://st.yandex-team.ru/AUDIENCE-163
    https://wiki.yandex-team.ru/jurijjgalickijj/audience/internal-dmp/
    https://tech.yandex.ru/audience/
    """

    def __init__(
        self,
        oauth_token=None,
        tvm_src_id=None,
        tvm_secret=None,
        n_retries=AUDIENCE_RETRIES,
        yt=None,
        logger=None,
        url=AUDIENCE_API_URL,
        port=AUDIENCE_API_PORT
    ):
        tvm_client = tvmauth.TvmClient(tvmauth.TvmApiClientSettings(
            self_tvm_id=tvm_src_id or config.UNICORN_AUDIENCE_TVM_ID,
            self_secret=tvm_secret or get_secrets().get_secret('UNICORN_AUDIENCE_TVM_SECRET'),
            dsts=[client.AUDIENCE_API_TVM_ID],
        ))

        super(AudienceClient, self).__init__(
            oauth_token=oauth_token,
            tvm_client=tvm_client,
            tvm_dst_id=client.AUDIENCE_API_TVM_ID,
            retries_settings=client.RetriesSettings(tries_count=n_retries or AUDIENCE_RETRIES),
            logger=logger or get_logger('audience_client'),
            url=url,
            port=port,
        )
        self.yt = yt or get_yt_client()

    def upload_segment_from_yt(self, ulogin, table, table_field, segment_name):
        n_rows = self.yt.row_count(table)
        self.logger.info('Going to download {} records from {}'.format(n_rows, table))

        with tempfile.NamedTemporaryFile(mode='w+b') as tempf:
            for i, row in enumerate(self.yt.read_table(self.yt.TablePath(table, columns=[table_field]))):
                if not (i % round(float(n_rows) * 0.05)):
                    self.logger.info('%.1f%%' % (100 * (i + 1) / float(n_rows)))
                tempf.write(str(row[table_field]) + '\n')
            tempf.seek(0)  # setting file offset to zero
            self.logger.info('Downloading finished!')

            return self.upload_segment_from_file(
                tempf.name,
                segment_name,
                content_type='yuid',
                hashed=False,
                ulogin=ulogin,
            )

    def modify_segment_with_data_from_yt(self, ulogin, table, table_field, segment_id, modification_type):
        n_rows = self.yt.row_count(table)
        self.logger.info('Going to download {} records from {}'.format(n_rows, table))

        with tempfile.NamedTemporaryFile(mode='w+b') as tempf:
            for i, row in enumerate(self.yt.read_table(self.yt.TablePath(table, columns=[table_field]))):
                if not (i % round(float(n_rows) * 0.05)):
                    self.logger.info('%.1f%%' % (100 * (i + 1) / float(n_rows)))
                tempf.write(str(row[table_field]) + '\n')
            tempf.seek(0)  # setting file offset to zero
            self.logger.info('Downloading finished!')

            return self.modify_segment_with_data_from_file(tempf.name, segment_id, modification_type, ulogin)

    def get_yuids_from_segment(self, segment_id):
        # use with tvm_src_id for crypta-profile-production service
        response = self._make_get_request(
            'http://audience-intapid.metrika.yandex.ru:9099/crypta/users',
            params={'segment_id': segment_id},
            headers=get_tvm_headers(
                self.tvm_client.get_service_ticket_for(tvm_id=client.AUDIENCE_INTAPI_TVM_ID)
            ),
        )

        yuid_list = response.text.split()[:-1]
        return yuid_list
