#!/usr/bin/env python
# -*- coding: utf-8 -*-

from crypta.lib.python.retryable_http_client import RetryableHttpClient
from crypta.profile.utils.yt_utils import get_yt_client
from crypta.profile.utils.config.secrets import get_secrets


CLICKHOUSE_HOST = 'http://clickhouse.metrika.yandex.net:8123/'
CLICKHOUSE_USER = 'robot-crypta'
CLICKHOUSE_CONNECTION_TIMEOUT = 5
CLICKHOUSE_RETRIES = 10


class ClickhouseClient(RetryableHttpClient):
    def __init__(self, host=CLICKHOUSE_HOST, user=CLICKHOUSE_USER,
                 password=None, timeout=CLICKHOUSE_CONNECTION_TIMEOUT, n_retries=CLICKHOUSE_RETRIES, logger=None):
        super(ClickhouseClient, self).__init__(n_retries, timeout)
        self.host = host
        self.user = user
        self.password = password or get_secrets().get_secret('CLICKHOUSE_PASSWORD')
        self.logger = logger

    def make_query(self, query):
        if self.logger:
            self.logger.info(query)
        params = {
            'user': self.user,
            'password': self.password,
            'skip_unavailable_shards': '1',
        }

        response = self._make_post_request(
            url=self.host,
            params=params,
            data=query,
        )

        return response.content

    @staticmethod
    def result_to_dict_list(result, fieldnames, extra_fields=None):
        if not result:
            return []
        records = []
        for line in result.splitlines():
            record = {fieldnames[i]: field for i, field in enumerate(line.split('\t'))}
            if extra_fields:
                record.update(extra_fields)
            records.append(record)
        return records

    @staticmethod
    def result_to_pandas(result, fieldnames):
        import pandas as pd
        records = ClickhouseClient.result_to_dict_list(result, fieldnames)
        dataframe = pd.DataFrame(records)
        return dataframe

    @staticmethod
    def result_to_yt(result, fieldnames, destination_table, yt=None):
        if not yt:
            yt = get_yt_client()
        records = ClickhouseClient.result_to_dict_list(result, fieldnames)
        yt.write_table(destination_table, records)

    def get_goal_reaches(self, start_date, end_date, counter_id, goal_id, min_duration=0, fields=None):
        goal_id = ','.join(goal_id) if type(goal_id) == list else goal_id
        fields = fields or ['UserID', 'StartDate', 'Duration', 'PageViews', 'Goals.ID']
        select_fields = ', '.join(fields)
        query = '''SELECT {select_fields}
                  FROM visits_all
                  WHERE
                      CounterID = {counter_id} AND
                      StartDate >= toDate('{start_date}') AND
                      StartDate <= toDate('{end_date}') AND
                      Duration >= {min_duration} AND
                      arrayExists(x -> x IN ({goal_id}), Goals.ID)
                  '''.format(select_fields=select_fields,
                             counter_id=counter_id,
                             goal_id=goal_id,
                             start_date=start_date,
                             end_date=end_date,
                             min_duration=min_duration)
        result = self.make_query(query)
        return result, fields

    def get_urls_like_hits(self, start_date, end_date, counter_id, urls_like_list, fields=None):
        fields = fields or ['UserID', 'EventTime', 'GoalsReached', 'URL', 'Referer']
        urls_query = ' OR '.join(["URL like '%s'" % url for url in urls_like_list])
        select_fields = ', '.join(fields)
        query = '''SELECT {select_fields}
                FROM hits_all
                WHERE
                    EventDate >= toDate('{start_date}') AND
                    EventDate <= toDate('{end_date}') AND
                    CounterID = {counter_id} AND
                    ({urls_query}) AND
                    NOT Refresh AND
                    NOT DontCountHits
                '''.format(select_fields=select_fields,
                           start_date=start_date,
                           end_date=end_date,
                           counter_id=counter_id,
                           urls_query=urls_query)
        result = self.make_query(query)
        return result, fields
