#!/usr/bin/env python
# -*- coding: utf-8 -*-
from collections import defaultdict

from cached_property import cached_property

from crypta.profile.utils.api import get_api
from crypta.profile.lib.segments import condition_helpers


class LabSegmentsInfo(object):
    def __init__(self):
        self.api = get_api(env='production')

    @cached_property
    def segments(self):
        return self.api.lab.getAllSegments().result()

    @cached_property
    def groups(self):
        return self.api.lab.getAllSegmentGroups().result()

    @cached_property
    def rules(self):
        return self.api.lab.getAllRules().result()

    @cached_property
    def segment_parents(self):
        return self.api.lab.getParentsPerSegment().result()

    @cached_property
    def segment_names(self):
        segment_id_to_name_dict = {}

        for segment in self.segments:
            segment_id_to_name_dict[segment.id] = segment.name

        return segment_id_to_name_dict

    def get_interest_ids(self, interest_type):
        lab_id_to_interest_id_dict = {}

        for segment in self.segments:
            for export in segment.exports.exports:
                if export.type == interest_type:
                    lab_id_to_interest_id_dict[segment.id] = export.segmentId

        return lab_id_to_interest_id_dict

    @cached_property
    def lab_segment_id_to_shortterm_interest_id(self):
        return self.get_interest_ids('SHORTTERM')

    @cached_property
    def lab_segment_id_to_longterm_interest_id(self):
        return self.get_interest_ids('LONGTERM')

    @cached_property
    def segment_only_segment_parents(self):
        segment_only_segment_parents = dict()

        for segment_id, segment_parents in self.segment_parents.iteritems():
            segment_parents_filtered = list(filter(lambda x: x.startswith('segment-'), segment_parents))
            if len(segment_parents_filtered) > 0:
                segment_only_segment_parents[segment_id] = segment_parents_filtered

        return segment_only_segment_parents

    def get_segment_with_segment_only_parents(self, segment_id):
        return [unicode(segment_id)] + self.segment_only_segment_parents.get(segment_id, list())

    def get_full_segment_name(self, segment_id):
        segment_name_parts = []
        for segment_parent_id in self.get_segment_with_segment_only_parents(segment_id)[::-1]:
            segment_name_parts.append(self.segment_names.get(segment_parent_id, '_'))

        return '/'.join(segment_name_parts)


def get_catalogia_conditions():
    api = get_api(env='production')

    rule_revision_id_to_catalogia_categories = {}
    rule_id_to_rule_revision_id = {}

    catalogia_conditions = api.lab.getAllRulesConditionsBySource(source='CATALOGIA').result()

    for condition in catalogia_conditions:
        if condition.state == 'APPROVED':
            categories = condition_helpers.normalize_categories(condition.values)
            if len(categories) == 0:
                continue
            rule_revision_id_to_catalogia_categories[condition.revision] = categories
            rule_id_to_rule_revision_id[condition.ruleId] = condition.revision

    return rule_revision_id_to_catalogia_categories, rule_id_to_rule_revision_id


def get_rule_id_to_catalogia_categories():
    rule_revision_id_to_catalogia_categories, rule_id_to_rule_revision_id = get_catalogia_conditions()

    rule_id_to_catalogia_categories = {}

    for rule_id, rule_revision_id in rule_id_to_rule_revision_id.iteritems():
        rule_id_to_catalogia_categories[rule_id] = rule_revision_id_to_catalogia_categories[rule_revision_id]

    return rule_id_to_catalogia_categories


def get_segments_with_shortterm_exports_with_rules():
    api = get_api(env='production')

    segment_id_to_rule_id = {}

    segments = api.lab.getAllSegments().result()

    for segment in segments:
        for export in segment.exports.exports:
            if export.type == 'SHORTTERM' and export.state not in ('DISABLED', 'DELETED') and export.ruleId:
                segment_id_to_rule_id[segment.id] = export.ruleId

    return segment_id_to_rule_id


def get_catalogia_interest_segment_ids_with_categories():
    rule_id_to_catalogia_categories = get_rule_id_to_catalogia_categories()
    segment_id_to_rule_id = get_segments_with_shortterm_exports_with_rules()

    segment_id_to_catalogia_categories = {}

    for segment_id, rule_id in segment_id_to_rule_id.iteritems():
        if rule_id in rule_id_to_catalogia_categories:
            segment_id_to_catalogia_categories[segment_id] = rule_id_to_catalogia_categories[rule_id]

    return segment_id_to_catalogia_categories


def get_catalogia_rules_for_interests():
    lab_segments_info = LabSegmentsInfo()

    rule_revision_id_to_catalogia_categories, rule_id_to_rule_revision_id = get_catalogia_conditions()
    segment_id_to_rule_id = get_segments_with_shortterm_exports_with_rules()

    rule_revision_id_to_segment_ids = {}
    for segment_id, rule_id in segment_id_to_rule_id.iteritems():
        rule_revision_id_to_segment_ids[rule_id_to_rule_revision_id[rule_id]] = lab_segments_info.get_segment_with_segment_only_parents(segment_id)

    catalogia_category_to_rule_revision_ids = defaultdict(set)
    for rule_revision_id, catalogia_categories in rule_revision_id_to_catalogia_categories.iteritems():
        if rule_revision_id not in rule_revision_id_to_segment_ids:
            continue

        for category in catalogia_categories:
            catalogia_category_to_rule_revision_ids[category].add(rule_revision_id)

    return catalogia_category_to_rule_revision_ids, rule_revision_id_to_segment_ids
