#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
import logging
import logging.handlers
import os
import sys
import time

import pathlib2

from crypta.profile.lib import date_helpers
from crypta.profile.utils.api import get_api
from crypta.profile.utils.config import config


def get_logger(name):
    logger = logging.getLogger(name)
    logger.setLevel(logging.DEBUG)
    console_handler = logging.StreamHandler(sys.stdout)
    formatter = logging.Formatter('%(asctime)s\t%(levelname)s\t%(message)s', '%Y-%m-%d %H:%M:%S')
    console_handler.setFormatter(formatter)
    logger.handlers = [console_handler]
    return logger


def get_file_logging_handler(log_file_path, logging_level=logging.INFO):
    formatter = logging.Formatter('%(asctime)s\t%(levelname)s\t%(process)s\t%(message)s', '%Y-%m-%d %H:%M:%S')
    pathlib2.Path(os.path.dirname(log_file_path)).mkdir(exist_ok=True, parents=True)

    file_handler = logging.handlers.RotatingFileHandler(log_file_path, maxBytes=16 * 1024 * 1024, backupCount=99)
    file_handler.setFormatter(formatter)
    file_handler.setLevel(logging_level)
    return file_handler


def get_file_logger(name, directory=None, filename=None):
    if not directory:
        directory = os.path.join(config.TASKS_LOGS_DIRECTORY, str(datetime.date.today()))
    log_file_path = os.path.join(directory, '{}.log'.format(filename or name))
    file_logging_handler = get_file_logging_handler(log_file_path)
    logger = logging.getLogger(name)
    logger.handlers = [file_logging_handler]
    logger.setLevel(logging.DEBUG)
    return logger, log_file_path


def get_stderr_logger():
    stderr_logger = logging.getLogger(__name__)
    log_handler = logging.StreamHandler(sys.stderr)
    stderr_logger.handlers = [log_handler]
    stderr_logger.setLevel(logging.DEBUG)
    return stderr_logger


logger = get_stderr_logger()


class TimeTracker(object):
    def __init__(self, monitoring_name):
        self.monitoring_name = monitoring_name

    def __enter__(self):
        self.start = time.time()

    def __exit__(self, *exc_info):
        end = time.time()
        time_spent = end - self.start
        logger.info('%s took %d seconds', self.monitoring_name, time_spent)
        send_to_graphite('timing.{}'.format(self.monitoring_name), time_spent)

        today_timestamp = date_helpers.from_date_string_to_timestamp(str(datetime.date.today()))

        send_to_graphite(
            'task_start.{}'.format(self.monitoring_name),
            int(self.start) - today_timestamp,
            timestamp=today_timestamp,
        )
        send_to_graphite(
            'task_end.{}'.format(self.monitoring_name),
            int(end) - today_timestamp,
            timestamp=today_timestamp,
        )


def send_to_graphite(name, value, group='crypta_classification_site2vec', timestamp=None):
    """Sends metrics via API.
    """
    hostname = 'crypta_profile_{}'.format(config.environment)
    effective_group = group.replace('.', '_')
    effective_timestamp = timestamp or int(time.time())
    try:
        reported_event = get_api(env='production').metric.report(
            name=name,
            hostname=hostname,
            group=effective_group,
            value=value,
            timestamp=effective_timestamp,
        ).result()
        logger.info('Reported %s', reported_event)
        return True
    except Exception:
        logger.exception('Failed to report metric to API: [%s.%s.%s %f]',
                         hostname, effective_group, name, value)
        return False
