#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os
import json
import logging

import luigi

import yt.logger as yt_logger
from yt.wrapper import TablePath, YtOperationFailedError

from crypta.lib.python.juggler.juggler_helpers import report_event_to_juggler
from crypta.lib.python.logging import logging_helpers
from crypta.profile.lib import date_helpers
from crypta.profile.utils.config import config
from crypta.profile.utils.loggers import (
    get_file_logger,
    get_file_logging_handler,
    get_stderr_logger,
)
from crypta.profile.utils.yql_utils import Yql, add_yql_logger_handler
from crypta.profile.utils.yt_utils import get_yt_client

logger = logging.getLogger(__name__)
logging_helpers.configure_stdout_logger(logger)


def get_significant_param_values(task):
    significant_param_names = sorted(task.get_param_names())
    significant_param_values = [str(getattr(task, key)).replace('/', '_') for key in significant_param_names]

    try:
        date_index = significant_param_names.index('date')
        date_param = significant_param_values[date_index]
        del significant_param_values[date_index]
        return date_param, significant_param_values
    except ValueError:
        pass

    return None, significant_param_values


def add_yt_logger_handler(log_file_path, logging_level=logging.DEBUG):
    yt_logger.LOGGER.setLevel(logging_level)
    yt_logger.LOGGER.handlers = [get_file_logging_handler(log_file_path, logging_level=logging_level)]


def configure_luigi_stdout_logger():
    luigi_logger = logging.getLogger("luigi-interface")
    logging_helpers.configure_stdout_logger(luigi_logger)


@luigi.Task.event_handler(luigi.Event.START)
def on_task_start(task):
    if hasattr(task, 'yql'):
        task.yql.yt = get_yt_client()

    if hasattr(task, 'log_file_path'):
        add_yt_logger_handler(task.log_file_path, logging_level=logging.INFO)
        if hasattr(task, 'yt'):
            task.logger.info('on_task_start: {} {}'.format(os.getpid(), task.yt))
        if hasattr(task, 'yql'):
            add_yql_logger_handler(task.log_file_path, logging_level=logging.INFO)


@luigi.Task.event_handler(luigi.Event.FAILURE)
def mourn_failure(task, exception):
    """Will be called directly after a failed execution
       of `run` on any JobTask subclass
    """
    if hasattr(task, 'logger'):
        if isinstance(exception, YtOperationFailedError):
            task.logger.error(json.dumps(exception.attributes['stderrs'], ensure_ascii=False))
        else:
            task.logger.exception(exception)

        task.logger.info('Task failed')

    task_group = getattr(task, 'task_group', 'default')
    juggler_host = getattr(task, 'juggler_host', config.CRYPTA_PROFILE_JUGGLER_HOST)

    task_name = task.__class__.__name__
    if getattr(task, 'other_significant_param_values', None):
        task_name += '_' + '_'.join(task.other_significant_param_values)

    juggler_status = 'CRIT' if task_group == 'export_profiles' else 'WARN'

    report_event_to_juggler(
        status=juggler_status,
        service=task_name,
        host=juggler_host,
        description=task_name,
        tags=[task_group],
        logger=get_stderr_logger(),
    )


@luigi.Task.event_handler(luigi.Event.SUCCESS)
def celebrate_success(task):
    if hasattr(task, 'logger'):
        task.logger.info('Task completed successfully')

    task_group = 'default'
    if hasattr(task, 'task_group'):
        task_group = task.task_group

    juggler_host = config.CRYPTA_PROFILE_JUGGLER_HOST
    if hasattr(task, 'juggler_host'):
        juggler_host = task.juggler_host

    task_name = task.__class__.__name__
    if getattr(task, 'other_significant_param_values', None):
        task_name += '_' + '_'.join(task.other_significant_param_values)

    report_event_to_juggler(
        status='OK',
        service=task_name,
        host=juggler_host,
        description='',
        tags=[task_group],
        logger=get_stderr_logger(),
    )


class BaseYtTask(luigi.Task):
    @property
    def yt(self):
        return get_yt_client()

    def __init__(self, *args, **kwargs):
        super(BaseYtTask, self).__init__(*args, **kwargs)
        date_param_value, other_significant_param_values = get_significant_param_values(self)
        self.other_significant_param_values = other_significant_param_values

        significant_param_values = other_significant_param_values
        if date_param_value is not None:
            significant_param_values = [date_param_value] + significant_param_values

        configured_logger, log_file_path = get_file_logger(
            name=self.__class__.__name__ + ''.join(significant_param_values),
            directory=os.path.join(config.TASKS_LOGS_DIRECTORY, *significant_param_values),
            filename=self.__class__.__name__,
        )
        self.logger = configured_logger
        self.log_file_path = log_file_path

        self.yql = Yql(logger=self.logger, yt=self.yt)


class BaseTimestampYtTask(luigi.Task):
    @property
    def yt(self):
        return get_yt_client()

    def __init__(self, timestamp):
        configured_logger, log_file_path = get_file_logger(
            name=self.__class__.__name__,
            directory=os.path.join(config.TIMESTAMP_TASKS_LOGS_DIRECTORY, str(timestamp)),
        )
        self.logger = configured_logger
        self.log_file_path = log_file_path

        self.yql = Yql(logger=self.logger, yt=self.yt)
        super(BaseTimestampYtTask, self).__init__(timestamp)


# targets and inputs


class YtTarget(luigi.Target):
    """
        To be used in case when generating tables with unique names. Like //some/path/2017-01-04
    """
    def __init__(self, table, allow_empty=False, yt_client=None, columns=None):
        self.external_yt_client = yt_client
        self.table = table
        self.allow_empty = allow_empty
        self.columns = columns
        if self.columns:
            self.table = TablePath(table, columns=columns)
        else:
            self.table = table

    @property
    def yt(self):
        # TODO(kolontaev): Отпилить к чертям external_yt_client
        if self.external_yt_client is not None:
            return self.external_yt_client

        return get_yt_client()

    def exists(self):
        if self.yt.exists(self.table):
            attributes = self.yt.get_table_attributes(self.table)
            return self.allow_empty or \
                attributes.get('row_count') or \
                attributes.get('chunk_row_count') or \
                attributes.get('type') != 'table'
        return False


class ExternalInput(luigi.ExternalTask):
    table = luigi.Parameter()
    columns = luigi.TupleParameter(default=())
    yt_client = luigi.Parameter(default=None)

    def output(self):
        return YtTarget(self.table, columns=list(self.columns), yt_client=self.yt_client)


class YtDailyRewritableTarget(luigi.Target):
    """
        To be used in case of updating table with the same name. Like //some/path/table_name
    """
    def __init__(self, table, date, allow_empty=False, allow_great_or_equal_date=False):
        self.table = table
        self.date = date
        self.allow_empty = allow_empty
        self.allow_great_or_equal_date = allow_great_or_equal_date

    @property
    def yt(self):
        return get_yt_client()

    def exists(self):
        # using custom attribute instead of modification time to ignore its change by merging and sorting
        generate_date = self.yt.get_attribute(self.table, 'generate_date', default=None)
        if self.yt.exists(self.table) \
                and (generate_date == self.date or (self.allow_great_or_equal_date and generate_date >= self.date)):
            if self.yt.get_attribute(self.table, 'type') == 'table':
                return self.allow_empty or bool(self.yt.get_attribute(self.table, 'row_count'))
            else:
                # do not check row count for map nodes and symlinks
                return True
        return False


class YtDateTarget(YtTarget):
    def __init__(self, table, date, field=None, allow_empty=False, columns=None):
        super(YtDateTarget, self).__init__(
            table,
            allow_empty=allow_empty,
            columns=columns,
        )
        self.date = date
        self.field = field

    def exists(self):
        yt_client = get_yt_client()
        if not super(YtDateTarget, self).exists():
            return False

        if self.field:
            generate_date = yt_client.get_attribute(
                self.table,
                self.field,
                None,
            )
        else:
            generate_date = yt_client.get_attribute(
                self.table,
                'generate_date',
                None,
            )
            if generate_date is None:
                generate_date = yt_client.get_attribute(
                    self.table,
                    'modification_time',
                )

        if generate_date is None:
            return False

        return self.date <= generate_date[:10]


class ExternalInputDate(luigi.ExternalTask):
    table = luigi.Parameter()
    date = luigi.Parameter()
    field = luigi.Parameter(default=None)
    columns = luigi.TupleParameter(default=())

    def output(self):
        return YtDateTarget(self.table, self.date, self.field, columns=list(self.columns))


class YtTableAttributeTarget(YtTarget):
    def __init__(self, table, attribute_name, attribute_value,
                 allow_empty=False, columns=None):
        super(YtTableAttributeTarget, self).__init__(
            table,
            allow_empty=allow_empty,
            columns=columns,
        )
        self.attribute_name = attribute_name
        self.attribute_value = attribute_value

    def exists(self):
        yt_client = get_yt_client()
        return super(YtTableAttributeTarget, self).exists() and \
            yt_client.get_attribute(self.table, self.attribute_name, None) == self.attribute_value


class YtTableMultipleAttributeTarget(YtTarget):
    def __init__(self, table, attribute_dict,
                 allow_empty=False, columns=None):
        super(YtTableMultipleAttributeTarget, self).__init__(
            table,
            allow_empty=allow_empty,
            columns=columns,
        )
        self.attribute_dict = attribute_dict

    def exists(self):
        yt_client = get_yt_client()
        return super(YtTableMultipleAttributeTarget, self).exists() and \
            all([yt_client.get_attribute(self.table, attribute_name, None) == attribute_value for attribute_name, attribute_value in self.attribute_dict.iteritems()])


class YtNodeAttributeTarget(luigi.Target):
    def __init__(self, path, attribute_name, attribute_value):
        self.path = path
        self.attribute_name = attribute_name
        self.attribute_value = attribute_value

    @property
    def yt(self):
        return get_yt_client()

    def exists(self):
        return self.yt.exists(self.path) \
            and self.yt.get_attribute(self.path, self.attribute_name, default=None) == self.attribute_value


class AttributeExternalInput(luigi.ExternalTask):
    table = luigi.Parameter()
    attribute_name = luigi.Parameter()
    attribute_value = luigi.Parameter()
    columns = luigi.TupleParameter(default=())

    def __init__(self, *args, **kwargs):
        super(AttributeExternalInput, self).__init__(*args, **kwargs)

    def output(self):
        return YtTableAttributeTarget(
            self.table,
            self.attribute_name,
            self.attribute_value,
            columns=list(self.columns),
        )


# cleaners
def remove_old_nodes(yt, logger, tables):
    for table_path in tables:
        try:
            yt.remove(table_path, recursive=True, force=True)
        except Exception as err:
            logger.exception(err.message)


def get_old_nodes(yt, folder, last_date):
    def is_old_node(table_path):
        table_name = os.path.basename(table_path)

        try:
            return table_name < last_date
        except ValueError:
            return False

    return filter(is_old_node, yt.list(folder, absolute=True))


class OldNodesByNameCleaner(luigi.Task):
    date = luigi.Parameter()
    folder = luigi.Parameter()
    lifetime = luigi.Parameter()
    date_format = luigi.Parameter(default='%Y-%m-%d')

    def run(self):
        yt_client = get_yt_client()
        last_date = date_helpers.get_date_from_past(self.date, int(self.lifetime))
        log_directory = os.path.join(config.TASKS_LOGS_DIRECTORY, str(self.date))
        logger, log_file_path = get_file_logger(name=self.__class__.__name__, directory=log_directory)

        remove_old_nodes(yt_client, logger, get_old_nodes(yt_client, self.folder, last_date))

    def output(self):
        return NoOldNodesByNameTarget(
            date=self.date,
            folder=self.folder,
            lifetime=self.lifetime,
            date_format=self.date_format,
        )


class NoOldNodesByNameTarget(luigi.Target):
    def __init__(self, date, folder, lifetime, date_format):
        self.folder = folder
        self.date_format = date_format
        self.last_date = date_helpers.get_date_from_past(date, int(lifetime))

    def exists(self):
        return len(get_old_nodes(get_yt_client(), self.folder, self.last_date)) == 0
