#!/usr/bin/env python
# -*- coding: utf-8 -*-

from abc import abstractmethod
import logging
import os

import luigi
import yt.logger as yt_logger

from crypta.profile.utils.config import config
from crypta.profile.lib.frozen_dict import FrozenDict
from crypta.profile.lib import date_helpers
from crypta.profile.utils.loggers import get_file_logging_handler
from crypta.profile.utils.luigi_utils import OldNodesByNameCleaner, YtDailyRewritableTarget, YtTarget
from crypta.profile.utils.yql_utils import Yql
from crypta.profile.utils.yt_utils import get_yt_client


class DayProcessor(luigi.Task):
    date = luigi.Parameter()

    def __init__(self, date):
        super(DayProcessor, self).__init__(date)

        # configure logging
        self.log_file_path = os.path.join(
            config.LOCAL_LOGS_DIRECTORY,
            'day_processors',
            date,
            '{}.log'.format(self.__class__.__name__),
        )

        self.logger = logging.getLogger(self.__class__.__name__)
        self.logger.handlers = [get_file_logging_handler(self.log_file_path)]
        self.logger.setLevel(logging.INFO)
        self.logger.info(
            'Init logger for {} handlers = {}'.format(
                self.__class__.__name__,
                self.logger.handlers,
            )
        )

        day_processor_output_folder = os.path.join(
            config.DAY_PROCESSORS_OUTPUT_FOLDER,
            self.__class__.__name__,
        )

        self.yql = Yql(logger=self.logger, yt=self.yt)
        self.yt.create_directory(day_processor_output_folder)

        self._day_processor_output = os.path.join(
            day_processor_output_folder,
            self.date,
        )

        self.transaction = None

    @property
    def output_directory(self):
        return os.path.join(
            config.DAY_PROCESSORS_OUTPUT_FOLDER,
            self.__class__.__name__,
        )

    @abstractmethod
    def process_day(self, inputs, output_path):
        pass

    def output(self):
        return YtTarget(self._day_processor_output)

    def run(self):
        with self.yt.Transaction() as transaction:
            self.transaction = transaction
            self.process_day(self.input(), self._day_processor_output)

    @property
    def yt(self):
        return get_yt_client(pool=config.SEGMENTS_POOL)


class LogProcessor(luigi.Task):
    task = luigi.TaskParameter()
    date = luigi.Parameter()
    n_days = luigi.Parameter()
    parent_task_name = luigi.Parameter(default=None)
    task_params = luigi.Parameter(default=FrozenDict())

    def __init__(self, task, date, n_days, parent_task_name=None, task_params=None):
        super(LogProcessor, self).__init__(task, date, n_days, parent_task_name, task_params)
        self.task_params = self.task_params or FrozenDict()
        # configure logging
        self.log_file_path = os.path.join(
            config.LOCAL_LOGS_DIRECTORY,
            'log_processors',
            date,
            '{}.log'.format(self.task.__name__),
        )

        self.logger = logging.getLogger(self.__class__.__name__)
        self.logger.handlers = [get_file_logging_handler(self.log_file_path)]
        self.logger.setLevel(logging.INFO)
        self.logger.info(
            'Init logger for {} handlers = {}'.format(
                self.__class__.__name__,
                self.logger.handlers,
            )
        )

        self.tasks_to_run = []
        for date in date_helpers.generate_back_dates(self.date, self.n_days):
            self.tasks_to_run.append(self.task(date, **self.task_params))

        self._day_processor_output_folder = self.tasks_to_run[0].output_directory

        self._log_processor_output = os.path.join(
            config.LOG_PROCESSORS_OUTPUT_FOLDER,
            os.path.basename(self._day_processor_output_folder),
        )
        self.yql = Yql(logger=self.logger, yt=self.yt)

    @property
    def yt(self):
        return get_yt_client(pool=config.SEGMENTS_POOL)

    def requires(self):
        return {
            'DayProcessorTasks': self.tasks_to_run,
            'CleanOldDayProcessors': OldNodesByNameCleaner(
                date=self.date,
                folder=self._day_processor_output_folder,
                lifetime=self.n_days,
                date_format='%Y-%m-%d',
            )
        }

    def output(self):
        return YtDailyRewritableTarget(
            self._log_processor_output,
            self.date,
        )

    def run(self):
        input_tables = []

        for target in self.input()['DayProcessorTasks']:
            input_tables.append(target.table)

        with self.yt.Transaction():
            self.yt.run_merge(
                input_tables,
                self._log_processor_output,
                mode='unordered',
                spec={'combine_chunks': True},
            )

            self.yt.set_attribute(
                self.output().table,
                'generate_date',
                self.date,
            )


def redefine_logger_and_yt_client(task):
    task.yql.yt = get_yt_client(pool=config.SEGMENTS_POOL)
    if hasattr(task, 'logger'):
        task.logger.info('on_task_start: {} {}'.format(os.getpid(), task.yt))
        yt_logger.LOGGER = task.logger


@DayProcessor.event_handler(luigi.Event.START)
def on_task_start_day_processor(task):
    redefine_logger_and_yt_client(task)


@LogProcessor.event_handler(luigi.Event.START)
def on_task_start_log_processor(task):
    redefine_logger_and_yt_client(task)
