#!/usr/bin/env python
# -*- coding: utf-8 -*-
import datetime
from collections import Counter

from crypta.profile.lib import date_helpers


MIN_AGE = 5
MAX_AGE = 90


def calculate_segment_scores(sources_by_segments):
    segment_scores = Counter()
    segment_sources_dict = dict()

    for segment, source_list in sources_by_segments.iteritems():
        for source_info in source_list:
            if source_info['source'] in segment_sources_dict:
                if segment != segment_sources_dict[source_info['source']]['segment']:
                    # conflict
                    segment_sources_dict[source_info['source']]['weight'] = -1
                elif segment_sources_dict[source_info['source']]['weight'] != -1:
                    # confirmation from one source
                    segment_sources_dict[source_info['source']]['weight'] = \
                        max(segment_sources_dict[source_info['source']]['weight'], source_info['weight'])
            else:
                # new source
                segment_sources_dict[source_info['source']] = {
                    'segment': segment,
                    'weight': source_info['weight'],
                }

    for source in segment_sources_dict:
        if segment_sources_dict[source]['weight'] > 0:
            segment_scores[segment_sources_dict[source]['segment']] += \
                segment_sources_dict[source]['weight']

    if not segment_scores:
        segment_scores = None
    return segment_scores


socdem_storage_schema = {
    'id': 'string',
    'id_type': 'string',
    'source': 'string',
    'update_time': 'uint64',
    'gender': 'string',
    'income_segment': 'string',

    'year_of_birth': 'uint64',
    'age': 'uint64',
    'age_segment': 'string',
    'birth_date': 'string',
}


def get_year_from_birth_date(birth_date, date_format=date_helpers.DATE_FORMAT):
    parsed_dt = date_helpers.from_date_string_to_datetime(birth_date, date_format)
    return parsed_dt.year


def is_valid_birth_date(date_str, current_year, date_format=date_helpers.DATE_FORMAT):
    try:
        parsed_dt = date_helpers.from_date_string_to_datetime(date_str, date_format)
        return current_year - MAX_AGE <= parsed_dt.year <= current_year - MIN_AGE
    except ValueError:
        return False


# TODO(ermolmak): should be updated (with tests) to return None if out of range
def get_age_segment_from_age(age):
    if age < 18:
        return '0_17'
    elif age < 25:
        return '18_24'
    elif age < 35:
        return '25_34'
    elif age < 45:
        return '35_44'
    elif age < 55:
        return '45_54'
    else:
        return '55_99'


# TODO(ermolmak): pass date as parameter (CRYPTA-15115)
def get_age_segment_from_year_of_birth(year_of_birth):
    now = datetime.datetime.now()
    age = now.year - year_of_birth

    if MIN_AGE <= age <= MAX_AGE:
        return get_age_segment_from_age(age)

    return None


# TODO(ermolmak): pass date as parameter (CRYPTA-15115)
def get_age_segment_from_birth_date(birth_date):
    today = datetime.date.today()
    year, month, day = birth_date.split('-')

    age = today.year - int(year) - ((today.month, today.day) < (int(month), int(day)))
    if MIN_AGE <= age <= MAX_AGE:
        return get_age_segment_from_age(age)

    return None
