import logging
import os

from crypta.lib.python import script_name
import crypta.lib.python.sandbox.client as sandbox
import crypta.lib.python.yql.client as yql
from crypta.profile.utils.loggers import get_file_logging_handler
from crypta.profile.utils.config import config
from crypta.profile.utils.config.secrets import get_secrets
from crypta.profile.utils.loggers import get_stderr_logger
from crypta.profile.utils.yt_utils import get_yt_client


class YqlError(Exception):
    pass


def add_yql_logger_handler(log_file_path, logging_level=logging.DEBUG):
    standard_yql_logger = logging.getLogger('yql.client.request')
    standard_yql_logger.setLevel(logging.INFO)
    standard_yql_logger.handlers = [get_file_logging_handler(log_file_path, logging_level=logging_level)]

    yql.logger.setLevel(logging_level)
    yql.logger.handlers = [get_file_logging_handler(log_file_path, logging_level=logging_level)]


class Yql(object):
    def __init__(self, logger=None, yt=None, token=None):
        self.logger = logger or get_stderr_logger()
        self.yt = yt or get_yt_client()
        self.token = token or get_secrets().get_secret('YQL_TOKEN')
        self.yql_client = yql.create_yql_client(
            config.CRYPTA_YT_PROXY,
            token=self.token,
            pool=self.yt.config['spec_defaults']['pool'],
            tmp_folder=config.PROFILES_YQL_TMP_YT_DIRECTORY,
        )

    def query(
        self,
        query_string,
        transaction=None,
        udf_resource_dict=None,
        udf_url_dict=None,
        pool=None,
        title=None,
        syntax_version=1,
        tmp_folder=None,
        erasure_codec='lrc_12_2_2',
        compression_codec='brotli_3',
        yql_libs=None,
        attached_files=None,
    ):
        if not title:
            title = script_name.detect_script_name()

        udfs = []

        if udf_url_dict:
            assert isinstance(udf_url_dict, dict)
            udfs += [yql.Udf(udf_so_name, udf_url) for udf_so_name, udf_url in udf_url_dict.iteritems()]

        if udf_resource_dict:
            assert isinstance(udf_resource_dict, dict)
            sandbox_client = sandbox.SandboxClient()

            for udf_so_name, udf_resource in udf_resource_dict.iteritems():
                resource_id = sandbox_client.get_last_released_resource_id(udf_resource, "stable")
                url = "https://proxy.sandbox.yandex-team.ru/{}".format(resource_id)
                udfs.append(yql.Udf(udf_so_name, url))

        return self.yql_client.execute(
            query_string,
            syntax_version=syntax_version,
            title='YQL: {}'.format(title),
            transaction=transaction.transaction_id if transaction is not None else None,
            pool=pool,
            tmp_folder=tmp_folder,
            erasure_codec=erasure_codec,
            compression_codec=compression_codec,
            udfs=udfs,
            yql_libs=yql_libs,
            attached_files=attached_files,
        )


def query(
    query_string,
    transaction=None,
    pool=None,
    logger=None,
    yt=None,
    udf_resource_dict=None,
    udf_url_dict=None,
    yql_token=None,
    title=None,
    syntax_version=1,
    tmp_folder=None,
    yql_libs=None,
    attached_files=None,
):
    yql_client = Yql(logger=logger, yt=yt, token=yql_token)
    return yql_client.query(
        query_string,
        transaction=transaction,
        udf_resource_dict=udf_resource_dict,
        udf_url_dict=udf_url_dict,
        pool=pool,
        title=title,
        syntax_version=syntax_version,
        tmp_folder=tmp_folder,
        yql_libs=yql_libs,
        attached_files=attached_files,
    )


def get_udf_path(path):
    return "yt://{}/{}".format(
        os.getenv("CRYPTA_TEST_YQL_DB", config.UDF_CLUSTER),
        path
    )
