#include <ads/bsyeti/caesar/libs/profiles/proto/mobile_app.pb.h>
#include <ads/bsyeti/libs/mobile_app_categories/builder.h>
#include <ads/bsyeti/libs/mobile_app_categories/reader.h>
#include <ads/bsyeti/libs/mobile_apps/hash.h>

#include <crypta/lib/native/proto_serializer/proto_serializer.h>
#include <crypta/rmp/make_mobile_app_categories_base/app_categories.pb.h>

#include <mapreduce/yt/interface/client.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/logger/log.h>
#include <util/generic/scope.h>
#include <util/generic/vector.h>
#include <util/folder/path.h>
#include <util/system/fs.h>

namespace {
    using TBaseEditor = NBSYeti::NMobileAppCategories::TBaseEditor;

    bool Process(const TAppCategories& row, TBaseEditor& base) {
        const auto& hash = NBSYeti::NMobileApps::GetMd5Hash(static_cast<NBSYeti::NMobileApps::EPlatform>(row.GetSourceID()), row.GetBundleId());

        const auto& appData = NCrypta::NProtoSerializer::CreateFromString<NCSR::TMobileAppData>(row.GetData());

        const auto& categories = appData.GetBMCategories();

        typename decltype(base.Data)::mapped_type value;
        value.Categories = {categories.begin(), categories.end()};

        base.Data.emplace(hash, std::move(value));
        return true;
    }

    TBaseEditor ReadTable(const NYT::IClientPtr& client) {
        const auto path = NYT::TRichYPath("//home/bigb/caesar/stable/MobileAppsDict").Schema(NYT::CreateTableSchema<TAppCategories>());

        INFO_LOG << "loading applications from " << path.Path_ << "...\n";

        size_t malformed{0};

        TBaseEditor base;

        for (auto reader = client->CreateTableReader<TAppCategories>(path); reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();

            if (!Process(row, base)) {
                DEBUG_LOG << "malformed value was skipped: " << row.GetBundleId() << ", " << row.GetSourceID() << ", " << row.GetData() << "\n";
                ++malformed;
            }
        }

        INFO_LOG << "done, malformed=" << malformed << "\n";
        return base;
    }

    struct TOpts {
        TString OutputFolder;
        size_t Attempts = 3;
        bool Verbose = false;
    };

    TOpts GetOpts(int argc, const char** argv) {
        TOpts result;

        NLastGetopt::TOpts opts;
        opts.AddHelpOption('h');
        opts.AddLongOption("attempts", "Number of attempts before fail")
            .StoreResult(&result.Attempts);
        opts.AddLongOption("verbose", "Enable verbose logging")
            .NoArgument()
            .OptionalValue("1")
            .StoreResult(&result.Verbose);
        opts.AddLongOption("output", "Output folder")
            .Required()
            .StoreResult(&result.OutputFolder);

        NLastGetopt::TOptsParseResult res(&opts, argc, argv);
        return result;
    }

    TBaseEditor ReadBase(size_t attempts) {
        static const TVector<TString> clusters({"seneca-sas", "seneca-vla", "seneca-man"});

        for (const auto& cluster: clusters) {
            auto client = NYT::CreateClient(cluster);
            for (size_t attempt = 0; attempt < attempts; ++attempt) {
                try {
                    return ReadTable(client);
                } catch (NYT::TIOException& e) {
                    WARNING_LOG << "cannot load data from yt: " << e.what() << "\n";
                    ::usleep(2000u);
                } catch (...) {
                    ERROR_LOG << "cannot load data from yt: " << CurrentExceptionMessage() << "\n";
                    std::exit(-1);
                }
            }
        }

        ERROR_LOG << "cannot load data from yt\n";
        std::exit(-1);
    }

    void SaveBase(const TBaseEditor& base, const TString& filename) {
        try {
            INFO_LOG << "save to " << filename << "\n";
            NBSYeti::NMobileAppCategories::SaveToFile(base, filename);
            INFO_LOG << "done\n";
        } catch (...) {
            ERROR_LOG << "cannot save base: " << CurrentExceptionMessage() << "\n";
            std::exit(-2);
        }
    }
}

int main(int argc, const char** argv) {
    const auto& opts = GetOpts(argc, argv);

    DoInitGlobalLog(CreateLogBackend("cerr", opts.Verbose ? ELogPriority::TLOG_DEBUG : ELogPriority::TLOG_INFO, true));

    NFs::MakeDirectoryRecursive(opts.OutputFolder);

    const auto& base = ReadBase(opts.Attempts);
    const auto& filename = JoinFsPaths(opts.OutputFolder, "mobile_app_categories_v1.vinyl");

    SaveBase(base, filename);

    return 0;
}
