#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
import functools
import logging

from crypta.lib.python.logging import logging_helpers
from crypta.profile.lib import date_helpers
from crypta.rt_socdem.lib.python.model import utils
from crypta.rt_socdem.services.training.lib import (
    age_gender_income,
    catboost_train_sample,
    distribution,
    features_mapping,
    raw_train_sample,
    roc_auc,
    thresholds,
)

import nirvana.job_context as nv


logger = logging.getLogger(__name__)


def main():
    logging_helpers.configure_stderr_logger(logging.getLogger(), level=logging.INFO)
    logger.info('Realtime socdem training')

    context = nv.context()
    parameters = context.parameters
    inputs = context.get_inputs()
    outputs = context.get_outputs()
    json_input = inputs.get('json_input') if inputs.has('json_input') else None
    result_output = outputs.get('result')

    yt_client = utils.get_yt_client(nv_params=parameters)
    yql_client = utils.get_yql_client(nv_params=parameters)

    today = str(datetime.date.today())
    yesterday = date_helpers.get_yesterday(today)

    tasks_dict = {
        'separate_age_gender_income': functools.partial(
            age_gender_income.separate,
            yt_client=yt_client,
            yql_client=yql_client,
        ),
        'prepare_catboost_train_sample': functools.partial(
            catboost_train_sample.prepare,
            yt_client=yt_client,
        ),
        'calculate_features_mapping': functools.partial(
            features_mapping.compute,
            yt_client=yt_client,
            yql_client=yql_client,
            output=result_output,
            yesterday=yesterday,
        ),
        'find_thresholds': functools.partial(
            thresholds.find,
            yt_client=yt_client,
            yql_client=yql_client,
            json_local_file_output=result_output,
            date_for_metrics=today,
        ),
        'get_raw_train_sample': functools.partial(
            raw_train_sample.get,
            yt_client=yt_client,
            yql_client=yql_client,
            yesterday=yesterday,
        ),
        'get_roc_auc_gender': functools.partial(
            roc_auc.get,
            yt_client=yt_client,
            json_input=json_input,
            socdem='gender',
        ),
        'get_roc_auc_age': functools.partial(
            roc_auc.get,
            yt_client=yt_client,
            json_input=json_input,
            socdem='age',
        ),
        'get_roc_auc_income': functools.partial(
            roc_auc.get,
            yt_client=yt_client,
            json_input=json_input,
            socdem='income',
        ),
        'get_thresholds_pool': functools.partial(
            thresholds.get_pool,
            yt_client=yt_client,
            yql_client=yql_client,
            yesterday=yesterday,
        ),
        'send_distribution_metrics': functools.partial(
            distribution.send_metrics,
            yt_client=yt_client,
            yql_client=yql_client,
            date_for_metrics=today,
        ),
    }

    job_name = parameters.get('job_name')
    logger.info('Job name: {}'.format(job_name))

    if job_name in tasks_dict:
        tasks_dict[job_name]()
    else:
        logger.warn('Unknown job_name "{}"'.format(job_name))
        exit(1)


if __name__ == '__main__':
    main()
