import collections

from grut.libs.object_api.proto import object_api_service_pb2
from grut.libs.proto.objects.autogen import schema_pb2

from crypta.lib.python import proto


class ConversionSourceClient:
    object_type = schema_pb2.OT_CONVERSION_SOURCE
    SelectResponse = collections.namedtuple("SelectResponse", ["continuation_token", "data"])

    def __init__(self, object_api_client, logger):
        self.object_api_client = object_api_client
        self.logger = logger

    def update_bulk_uniformly(self, conversion_sources, set_paths):
        self.logger.debug("conversion_sources=%s; set_paths=%s;", conversion_sources, set_paths)

        request = object_api_service_pb2.TReqUpdateObjects(
            object_type=self.object_type,
            create_nonexistent=False,
        )

        for conversion_source in conversion_sources:
            payload = request.payloads.add(set_paths=set_paths)
            payload.meta = conversion_source.meta.SerializeToString()
            payload.spec = conversion_source.spec.SerializeToString()

        return self.object_api_client.update_objects(request)

    def select(
        self,
        attribute_selector=None,
        filter_=None,
        limit=1,
        continuation_token=None,
    ):
        request = object_api_service_pb2.TReqSelectObjects(
            object_type=self.object_type,
            filter=filter_ or "true",
            attribute_selector=attribute_selector or ["/meta", "/spec"],
            limit=limit,
            continuation_token=continuation_token,
        )

        self.logger.debug(request)

        response = self.object_api_client.select_objects(request)

        data = [
            proto.from_string(schema_pb2.TConversionSource, payload.protobuf)
            for payload in response.payloads
        ]

        return self.SelectResponse(response.continuation_token, data)

    def select_all(
        self,
        attribute_selector=None,
        filter_=None,
        limit=1,
    ):
        for batch in self.select_all_batched(filter_=filter_, limit=limit, attribute_selector=attribute_selector):
            for item in batch:
                yield item

    def select_all_batched(
        self,
        attribute_selector=None,
        filter_=None,
        limit=1,
    ):
        continuation_token = None
        while (response := self.select(attribute_selector, filter_, limit, continuation_token=continuation_token)).data:
            continuation_token = response.continuation_token
            yield response.data

    def get(self, conversion_source_id, attribute_selector):
        filter_ = f"[/meta/id] = {conversion_source_id}"
        response = self.select(attribute_selector=attribute_selector, filter_=filter_, limit=1)
        return response.data[0] if response.data else None
